<?php

class Facturacion_Electronica_Model extends Model
{
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Guardar documento de facturación electrónica
     */
    public function guardar_documento_fe($data)
    {
        $log_file = __DIR__ . '/../debug_facturacion_electronica.log';
        $log_message = date('Y-m-d H:i:s') . " === GUARDAR_DOCUMENTO_FE - INICIO ===\n";
        $log_message .= "Datos recibidos completos:\n" . print_r($data, true) . "\n";
        $log_message .= "Campos específicos de facturación electrónica:\n";
        $log_message .= "  - id_serie_electronica: " . (isset($data['id_serie_electronica']) ? $data['id_serie_electronica'] : 'NO DEFINIDO') . "\n";
        $log_message .= "  - serie_electronica: " . (isset($data['serie_electronica']) ? $data['serie_electronica'] : 'NO DEFINIDO') . "\n";
        $log_message .= "  - correlativo_electronico: " . (isset($data['correlativo_electronico']) ? $data['correlativo_electronico'] : 'NO DEFINIDO') . "\n";
        $log_message .= "  - pos_electronico: " . (isset($data['pos_electronico']) ? $data['pos_electronico'] : 'NO DEFINIDO') . "\n";
        $log_message .= "  - tipo_documento_electronico: " . (isset($data['tipo_documento_electronico']) ? $data['tipo_documento_electronico'] : 'NO DEFINIDO') . "\n";
        $log_message .= "  - tipo_doc: " . (isset($data['tipo_doc']) ? $data['tipo_doc'] : 'NO DEFINIDO') . "\n";
        $log_message .= "  - id_venta: " . (isset($data['id_venta']) ? $data['id_venta'] : 'NO DEFINIDO') . "\n";
        file_put_contents($log_file, $log_message, FILE_APPEND);
        error_log($log_message);
        
        try 
        {
            date_default_timezone_set($_SESSION["zona_horaria"]);
            $fecha = date("Y-m-d H:i:s");
            
            // Obtener datos de la serie de documento desde tm_tipo_doc
            $serie_data = null;
            if (isset($data['id_serie_electronica']) && !empty($data['id_serie_electronica'])) {
                // Verificar si es un id_tipo_doc (1, 2, 3) o un id_serie de tm_series_documentos
                $id_serie_electronica = intval($data['id_serie_electronica']);
                
                if ($id_serie_electronica <= 10) {
                    // Es un id_tipo_doc, obtener desde tm_tipo_doc
                    $stm_tipodoc = $this->db->prepare("SELECT * FROM tm_tipo_doc WHERE id_tipo_doc = ? AND estado = 'a'");
                    $stm_tipodoc->execute(array($id_serie_electronica));
                    $tipodoc_data = $stm_tipodoc->fetch(PDO::FETCH_ASSOC);
                    
                    if ($tipodoc_data) {
                        // Convertir datos de tm_tipo_doc al formato esperado
                        $serie_data = array(
                            'id_serie' => $tipodoc_data['id_tipo_doc'],
                            'tipo_documento' => isset($data['tipo_documento_electronico']) ? $data['tipo_documento_electronico'] : 'FAC',
                            'serie' => $tipodoc_data['serie'],
                            'pos' => $tipodoc_data['pos'] ?: '001',
                            'numero_inicial' => $tipodoc_data['numero_inicial'] ?: 1,
                            'numero_actual' => $tipodoc_data['numero_actual'] ?: ($tipodoc_data['numero'] ? intval($tipodoc_data['numero']) : 1),
                            'numero_final' => $tipodoc_data['numero_final'] ?: 9999999999,
                            'ambiente' => $tipodoc_data['ambiente'] ?: 2,
                            'activo' => $tipodoc_data['estado'] == 'a' ? 1 : 0
                        );
                    }
                } else {
                    // Es un id_serie real de tm_series_documentos (método antiguo)
                    $stm_serie = $this->db->prepare("SELECT * FROM tm_series_documentos WHERE id_serie = ?");
                    $stm_serie->execute(array($data['id_serie_electronica']));
                    $serie_data = $stm_serie->fetch(PDO::FETCH_ASSOC);
                }
            }
            
            // Obtener datos del cliente
            $cliente_data = null;
            if (isset($data['cliente_id']) && !empty($data['cliente_id'])) {
                $stm_cliente = $this->db->prepare("SELECT * FROM v_clientes WHERE id_cliente = ?");
                $stm_cliente->execute(array($data['cliente_id']));
                $cliente_data = $stm_cliente->fetch(PDO::FETCH_ASSOC);
                
                // Log de datos del cliente obtenidos
                $log_cliente = date('Y-m-d H:i:s') . " guardar_documento_fe: Cliente ID=" . $data['cliente_id'] . "\n";
                if ($cliente_data) {
                    $log_cliente .= "  - nombre: " . (isset($cliente_data['nombre']) ? $cliente_data['nombre'] : 'N/A') . "\n";
                    $log_cliente .= "  - tipo_cliente: " . (isset($cliente_data['tipo_cliente']) ? $cliente_data['tipo_cliente'] : 'N/A') . "\n";
                    $log_cliente .= "  - ruc: " . (isset($cliente_data['ruc']) ? $cliente_data['ruc'] : 'N/A') . "\n";
                    $log_cliente .= "  - direccion: " . (isset($cliente_data['direccion']) ? $cliente_data['direccion'] : 'N/A') . "\n";
                    $log_cliente .= "  - ubicacion: " . (isset($cliente_data['ubicacion']) ? $cliente_data['ubicacion'] : 'N/A') . "\n";
                    $log_cliente .= "  - Campos disponibles: " . implode(', ', array_keys($cliente_data)) . "\n";
                } else {
                    $log_cliente .= "  - ERROR: Cliente no encontrado en v_clientes\n";
                }
                file_put_contents($log_file, $log_cliente, FILE_APPEND);
                error_log($log_cliente);
            }
            
            // Determinar tipo de receptor: usar el que viene del frontend si está disponible, sino inferirlo del tipo de cliente
            $receptor_type = '02'; // Por defecto consumidor final
            if (isset($data['receptor_type']) && !empty($data['receptor_type'])) {
                // Usar el tipo de receptor seleccionado en el frontend
                $receptor_type = trim($data['receptor_type']);
                // Validar que sea un tipo válido
                if (!in_array($receptor_type, array('01', '02', '03'))) {
                    $receptor_type = '02'; // Fallback a Consumidor Final si no es válido
                }
            } elseif ($cliente_data && $cliente_data['tipo_cliente'] == 2) {
                // Si no viene del frontend pero el cliente es contribuyente, usar tipo 01
                $receptor_type = '01';
            }
            
            $receptor_ruc = null;
            $receptor_ruc_type = null;
            $receptor_dv = null;
            $receptor_address = null;
            $receptor_location = null;
            
            // Validar y obtener datos según el tipo de receptor seleccionado
            if ($receptor_type == '01' || $receptor_type == '03') {
                // Contribuyente (01) o Gobierno (03): Requieren RUC, dirección y ubicación
                if (!$cliente_data || $cliente_data['tipo_cliente'] != 2) {
                    return array(
                        'success' => false, 
                        'message' => 'Para emitir a ' . ($receptor_type == '01' ? 'Contribuyente' : 'Gobierno') . ' se requiere un cliente con RUC registrado. Por favor, seleccione un cliente contribuyente.'
                    );
                }
                
                $receptor_ruc = $cliente_data['ruc'];
                if (empty($receptor_ruc)) {
                    return array(
                        'success' => false, 
                        'message' => 'El cliente seleccionado no tiene RUC registrado. Por favor, seleccione un cliente con RUC válido.'
                    );
                }
                
                $receptor_ruc_type = strlen($receptor_ruc) == 9 ? '1' : '2'; // 1=Natural, 2=Jurídico
                $receptor_address = $cliente_data['direccion'];
                
                // Validar que tenga dirección (requerida para contribuyentes y gobierno en Panamá)
                if (empty($receptor_address) || trim($receptor_address) == '') {
                    // Intentar obtener dirección desde los datos enviados
                    if (isset($data['direccion_cliente']) && !empty($data['direccion_cliente'])) {
                        $receptor_address = $data['direccion_cliente'];
                    } else {
                        // Si no hay dirección, lanzar error
                        return array(
                            'success' => false, 
                            'message' => 'El cliente seleccionado no tiene dirección registrada. Por favor, ingrese la dirección del cliente según las reglas de Panamá (mínimo 5 caracteres).'
                        );
                    }
                }
                
                // Validar formato de dirección (mínimo 5 caracteres según reglas de Panamá)
                if (strlen(trim($receptor_address)) < 5) {
                    return array(
                        'success' => false, 
                        'message' => 'La dirección del cliente debe tener al menos 5 caracteres según las reglas de Panamá.'
                    );
                }
                
                // Obtener ubicación si está disponible (requerida para contribuyentes y gobierno)
                $log_ubicacion = date('Y-m-d H:i:s') . " guardar_documento_fe: Buscando ubicación para cliente ID " . $data['cliente_id'] . ":\n";
                $log_ubicacion .= "  - cliente_data['ubicacion']: " . (isset($cliente_data['ubicacion']) ? var_export($cliente_data['ubicacion'], true) : 'NO EXISTE') . "\n";
                $log_ubicacion .= "  - cliente_data['location']: " . (isset($cliente_data['location']) ? var_export($cliente_data['location'], true) : 'NO EXISTE') . "\n";
                $log_ubicacion .= "  - data['ubicacion_cliente']: " . (isset($data['ubicacion_cliente']) ? var_export($data['ubicacion_cliente'], true) : 'NO EXISTE') . "\n";
                
                if (isset($cliente_data['ubicacion']) && !empty($cliente_data['ubicacion'])) {
                    $receptor_location = $cliente_data['ubicacion'];
                    $log_ubicacion .= "  - ✅ Ubicación obtenida desde cliente_data['ubicacion']: " . $receptor_location . "\n";
                } elseif (isset($cliente_data['location']) && !empty($cliente_data['location'])) {
                    $receptor_location = $cliente_data['location'];
                    $log_ubicacion .= "  - ✅ Ubicación obtenida desde cliente_data['location']: " . $receptor_location . "\n";
                } elseif (isset($data['ubicacion_cliente']) && !empty($data['ubicacion_cliente'])) {
                    $receptor_location = $data['ubicacion_cliente'];
                    $log_ubicacion .= "  - ✅ Ubicación obtenida desde data['ubicacion_cliente']: " . $receptor_location . "\n";
                } else {
                    // La ubicación es requerida para contribuyentes y gobierno
                    $log_ubicacion .= "  - ❌ ERROR: No se encontró ubicación en ningún campo\n";
                    file_put_contents($log_file, $log_ubicacion, FILE_APPEND);
                    error_log($log_ubicacion);
                    return array(
                        'success' => false, 
                        'message' => 'El cliente seleccionado no tiene ubicación registrada (formato PP-DD-CC). Por favor, ingrese la ubicación del cliente.'
                    );
                }
                file_put_contents($log_file, $log_ubicacion, FILE_APPEND);
                error_log($log_ubicacion);
                
                // TODO: Calcular dígito verificador del RUC
            } else {
                // Consumidor Final (02): Solo requiere nombre, NO debe incluir RUC, DV ni ubicación
                // Puede tener dirección opcional si se proporciona
                if (isset($data['direccion_cliente']) && !empty($data['direccion_cliente'])) {
                    $receptor_address = $data['direccion_cliente'];
                }
            }
            
            // Determinar tipo de documento según tipo_documento_electronico recibido
            // En Panamá, tanto boletas como facturas se emiten como "FAC" (Factura)
            // No hay diferenciación entre "boleta" y "factura" como en otros países
            // Los tipos válidos son: FAC=Factura, NTC=Nota Crédito, NTD=Nota Débito
            $document_type = 'FAC'; // Valor por defecto
            if (isset($data['tipo_documento_electronico']) && !empty($data['tipo_documento_electronico'])) {
                // Usar el tipo que viene del frontend (configurado en tm_tipo_doc)
                $document_type = strtoupper(trim($data['tipo_documento_electronico']));
                // Validar que sea un tipo válido para Panamá
                if (!in_array($document_type, array('FAC', 'NTC', 'NTD'))) {
                    // Si no es válido, usar FAC por defecto
                    $document_type = 'FAC';
                }
            }
            
            // Obtener ambiente de la serie
            $environment = isset($serie_data['ambiente']) ? $serie_data['ambiente'] : 2; // Por defecto pruebas
            
            // Insertar documento principal
            $sql = "INSERT INTO tm_facturacion_electronica_documentos (
                id_venta, id_pedido, id_serie_documento,
                environment, issue_date, pos,
                fd_number, document_type, sale_type, info, total, dest_country, payment_term,
                receptor_type, receptor_name, receptor_ruc, receptor_ruc_type, 
                receptor_address, receptor_email, receptor_dv, receptor_location,
                status
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
            
            // Obtener el número fiscal a usar
            $fd_number = 0;
            if (isset($data['correlativo_electronico']) && !empty($data['correlativo_electronico'])) {
                // Si viene un correlativo específico, usarlo (remover ceros a la izquierda para obtener el número real)
                $fd_number = intval($data['correlativo_electronico']);
            } else if ($serie_data) {
                // Si no viene correlativo, usar el actual de la serie
                $fd_number = intval($serie_data['numero_actual']);
            }
            
            // POS debe ser exactamente 3 dígitos (000-999)
            $pos_raw = isset($data['pos_electronico']) ? trim($data['pos_electronico']) : '001';
            // Asegurar que tenga exactamente 3 dígitos, rellenando con ceros a la izquierda
            $pos = str_pad(substr(preg_replace('/[^0-9]/', '', $pos_raw), 0, 3), 3, '0', STR_PAD_LEFT);
            $receptor_name = $cliente_data ? ($cliente_data['razon_social'] ?: $cliente_data['nombre']) : 'Consumidor Final';
            $receptor_email = $cliente_data ? $cliente_data['correo'] : null;
            $total = isset($data['total']) ? $data['total'] : 0;
            
            // Para compatibilidad: si viene id_serie_electronica pero es un id_tipo_doc, no guardarlo como id_serie_documento
            // porque tm_facturacion_electronica_documentos.id_serie_documento hace referencia a tm_series_documentos
            $id_serie_documento = null;
            if (isset($data['id_serie_electronica']) && !empty($data['id_serie_electronica'])) {
                $id_serie_electronica = intval($data['id_serie_electronica']);
                // Si es menor o igual a 10, es un id_tipo_doc, guardar null
                if ($id_serie_electronica > 10) {
                    // Es un id_serie real de tm_series_documentos
                    $id_serie_documento = $id_serie_electronica;
                }
            }
            
            $log_message = date('Y-m-d H:i:s') . " Datos preparados para INSERT en tm_facturacion_electronica_documentos:\n";
            $log_message .= "  - id_venta: " . $data['id_venta'] . "\n";
            $log_message .= "  - id_pedido: " . (isset($data['id_pedido']) ? $data['id_pedido'] : 'NULL') . "\n";
            $log_message .= "  - id_serie_documento: " . ($id_serie_documento ?? 'NULL') . "\n";
            $log_message .= "  - environment: " . $environment . "\n";
            $log_message .= "  - pos: " . $pos . "\n";
            $log_message .= "  - fd_number: " . $fd_number . "\n";
            $log_message .= "  - document_type: " . $document_type . "\n";
            $log_message .= "  - total: " . $total . "\n";
            $log_message .= "  - receptor_type: " . $receptor_type . "\n";
            $log_message .= "  - receptor_name: " . $receptor_name . "\n";
            $log_message .= "  - receptor_ruc: " . ($receptor_ruc ?? 'NULL') . "\n";
            file_put_contents($log_file, $log_message, FILE_APPEND);
            error_log($log_message);
            
            $stmt = $this->db->prepare($sql);
            $stmt->execute(array(
                $data['id_venta'],
                isset($data['id_pedido']) ? $data['id_pedido'] : null,
                $id_serie_documento, // null si viene de tm_tipo_doc, id_serie si viene de tm_series_documentos
                $environment,
                $fecha,
                $pos,
                $fd_number,
                $document_type,
                '1', // sale_type: 1=Giro de negocio
                isset($data['observacion']) ? $data['observacion'] : null,
                $total,
                'PA', // dest_country
                1, // payment_term: 1=Contado
                $receptor_type,
                $receptor_name,
                $receptor_ruc,
                $receptor_ruc_type,
                $receptor_address,
                $receptor_email,
                $receptor_dv,
                $receptor_location,
                'pending' // status
            ));
            
            $id_documento_fe = $this->db->lastInsertId();
            
            $log_message = date('Y-m-d H:i:s') . " ✅ INSERT exitoso. ID documento FE creado: " . $id_documento_fe . "\n";
            file_put_contents($log_file, $log_message, FILE_APPEND);
            error_log($log_message);
            
            // Obtener configuración de impuesto incluido y tasa de impuesto
            $impuesto_incluido = false;
            if (isset($data['impuesto_incluido'])) {
                $impuesto_incluido = ($data['impuesto_incluido'] == '1' || $data['impuesto_incluido'] == 1);
            } elseif (Session::get('impuesto_incluido')) {
                $impuesto_incluido = (Session::get('impuesto_incluido') == '1' || Session::get('impuesto_incluido') == 1);
            }
            
            $igv_rate = 0.07; // 7% por defecto (ITBMS)
            if (isset($data['igv']) && !empty($data['igv'])) {
                $igv_rate = floatval($data['igv']);
            } elseif (Session::get('igv')) {
                $igv_rate = floatval(Session::get('igv'));
            }
            
            $log_message = date('Y-m-d H:i:s') . " Configuración de impuestos:\n";
            $log_message .= "  - impuesto_incluido: " . ($impuesto_incluido ? 'SI' : 'NO') . "\n";
            $log_message .= "  - igv_rate: " . ($igv_rate * 100) . "%\n";
            file_put_contents($log_file, $log_message, FILE_APPEND);
            error_log($log_message);
            
            // Guardar items
            $total_items_calculado = 0;
            if (isset($data['items']) && is_array($data['items'])) {
                $this->guardar_items_fe($id_documento_fe, $data['items'], $impuesto_incluido, $igv_rate);
                
                // Calcular el total real sumando todos los items guardados
                $sql_items_sum = "SELECT SUM(total_item) as total_items FROM tm_facturacion_electronica_items WHERE id_documento_fe = ?";
                $stm_items_sum = $this->db->prepare($sql_items_sum);
                $stm_items_sum->execute(array($id_documento_fe));
                $result_items_sum = $stm_items_sum->fetch(PDO::FETCH_ASSOC);
                $total_items_calculado = floatval($result_items_sum['total_items']);
                
                error_log("guardar_documento_fe: Total items calculado desde BD: " . $total_items_calculado);
                error_log("guardar_documento_fe: Total recibido en data: " . $total);
                
                // Solo actualizar el total del documento con la suma de ítems cuando el total recibido es 0 o no viene.
                // Si el total recibido ya viene con descuento (desde tm_venta), no sobrescribir para no perder el descuento.
                $total_recibido = floatval($total);
                if ($total_recibido <= 0 && $total_items_calculado > 0) {
                    error_log("guardar_documento_fe: Total recibido es 0, actualizando a suma de ítems: " . $total_items_calculado);
                    $sql_update_total = "UPDATE tm_facturacion_electronica_documentos SET total = ? WHERE id_documento_fe = ?";
                    $stm_update_total = $this->db->prepare($sql_update_total);
                    $stm_update_total->execute(array($total_items_calculado, $id_documento_fe));
                    $total = $total_items_calculado;
                } elseif (abs($total_items_calculado - $total_recibido) > 0.01) {
                    error_log("guardar_documento_fe: ADVERTENCIA - Diferencia entre total ítems (" . $total_items_calculado . ") y total documento (" . $total_recibido . "). Se mantiene el total del documento (puede incluir descuento).");
                }
            }
            
            // Usar el total del documento (con descuento si aplica) para los pagos, no la suma de ítems
            $total_para_pagos = floatval($total);
            if ($total_para_pagos <= 0 && $total_items_calculado > 0) {
                $total_para_pagos = $total_items_calculado;
            }
            
            // Guardar pagos
            if (isset($data['payments']) && is_array($data['payments']) && count($data['payments']) > 0) {
                error_log("guardar_documento_fe: Preparando para guardar " . count($data['payments']) . " pagos");
                
                // Verificar que la suma de pagos coincida con el total del documento (con impuestos)
                $suma_pagos = 0;
                foreach ($data['payments'] as $payment) {
                    $suma_pagos += isset($payment['monto']) ? floatval($payment['monto']) : 0;
                }
                
                error_log("guardar_documento_fe: Suma de pagos recibidos: " . $suma_pagos);
                error_log("guardar_documento_fe: Total del documento (con impuestos): " . $total_para_pagos);
                
                // Si la suma de pagos no coincide con el total, ajustar el último pago
                $diferencia_pagos = abs($suma_pagos - $total_para_pagos);
                if ($diferencia_pagos > 0.01) {
                    error_log("guardar_documento_fe: ADVERTENCIA - Diferencia entre suma de pagos y total: " . $diferencia_pagos);
                    
                    if ($suma_pagos < $total_para_pagos) {
                        // Si falta dinero, agregar la diferencia al último pago
                        $diferencia_faltante = $total_para_pagos - $suma_pagos;
                        error_log("guardar_documento_fe: Agregando diferencia faltante ($diferencia_faltante) al último pago");
                        
                        // Agregar la diferencia al último pago
                        if (count($data['payments']) > 0) {
                            $ultimo_pago = &$data['payments'][count($data['payments']) - 1];
                            $monto_actual = isset($ultimo_pago['monto']) ? floatval($ultimo_pago['monto']) : 0;
                            $ultimo_pago['monto'] = $monto_actual + $diferencia_faltante;
                            error_log("guardar_documento_fe: Último pago ajustado de $monto_actual a " . $ultimo_pago['monto']);
                        }
                    } else {
                        // Si sobra dinero, reducir el último pago
                        $diferencia_sobrante = $suma_pagos - $total_para_pagos;
                        error_log("guardar_documento_fe: Reduciendo último pago en $diferencia_sobrante");
                        
                        if (count($data['payments']) > 0) {
                            $ultimo_pago = &$data['payments'][count($data['payments']) - 1];
                            $monto_actual = isset($ultimo_pago['monto']) ? floatval($ultimo_pago['monto']) : 0;
                            $ultimo_pago['monto'] = max(0, $monto_actual - $diferencia_sobrante);
                            error_log("guardar_documento_fe: Último pago ajustado de $monto_actual a " . $ultimo_pago['monto']);
                        }
                    }
                }
                
                $this->guardar_payments_fe($id_documento_fe, $data['payments']);
            } else {
                // Si no hay pagos, crear uno por defecto con el total correcto (con impuestos)
                error_log("guardar_documento_fe: No se encontraron pagos para guardar. Creando pago por defecto con total: " . $total_para_pagos);
                $pago_defecto = array(
                    array(
                        'tipo' => '1', // Efectivo
                        'id_tipo_pago' => null,
                        'monto' => $total_para_pagos,
                        'descripcion' => 'Pago en efectivo'
                    )
                );
                $this->guardar_payments_fe($id_documento_fe, $pago_defecto);
            }
            
            // Actualizar correlativo en la serie de documento (tm_tipo_doc o tm_series_documentos)
            if (isset($data['id_serie_electronica']) && !empty($data['id_serie_electronica'])) {
                $id_serie_electronica = intval($data['id_serie_electronica']);
                // Verificar si es un id_tipo_doc (1, 2, 3) o un id_serie real
                if ($id_serie_electronica <= 10) {
                    // Es un id_tipo_doc, actualizar tm_tipo_doc
                    $this->actualizar_correlativo_tipodoc($id_serie_electronica, $fd_number);
                } else {
                    // Es un id_serie real de tm_series_documentos
                    $this->actualizar_correlativo_serie($id_serie_electronica, $fd_number);
                }
            }
            
            $log_message = date('Y-m-d H:i:s') . " ✅ Documento FE guardado completamente. ID: " . $id_documento_fe . "\n";
            $log_message .= "=== GUARDAR_DOCUMENTO_FE - FIN EXITOSO ===\n";
            file_put_contents($log_file, $log_message, FILE_APPEND);
            error_log($log_message);
            
            return array('success' => true, 'id_documento_fe' => $id_documento_fe);
            
        } catch (Exception $e) {
            $log_message = date('Y-m-d H:i:s') . " ❌ ERROR al guardar documento FE: " . $e->getMessage() . "\n";
            $log_message .= "Stack trace: " . $e->getTraceAsString() . "\n";
            $log_message .= "=== GUARDAR_DOCUMENTO_FE - FIN CON ERROR ===\n";
            file_put_contents($log_file, $log_message, FILE_APPEND);
            error_log($log_message);
            return array('success' => false, 'message' => $e->getMessage());
        }
    }
    
    /**
     * Guardar items del documento
     */
    private function guardar_items_fe($id_documento_fe, $items, $impuesto_incluido = false, $igv_rate = 0.07)
    {
        try {
            // Los parámetros ya vienen del método llamador, no necesitamos obtenerlos de sesión aquí
            
            $line = 1;
            foreach ($items as $item) {
                $sql_item = "INSERT INTO tm_facturacion_electronica_items (
                    id_documento_fe, id_producto, line, price, mu, quantity, 
                    description, internal_code, discount, subtotal, total_taxes, total_item
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
                
                $price = isset($item['precio']) ? floatval($item['precio']) : 0;
                $quantity = isset($item['cantidad']) ? floatval($item['cantidad']) : 1;
                $discount = isset($item['descuento']) ? floatval($item['descuento']) : 0;
                
                // Misma lógica que imp_venta_all: precio unitario real, descuento por ítem, monto = (cant*precio)-descuento
                // subtotal_linea = cantidad * precio_unitario; monto_linea = subtotal_linea - descuento_item
                if ($impuesto_incluido) {
                    $precio_con_impuesto = ($price * $quantity) - $discount;
                    $subtotal = $precio_con_impuesto / (1 + $igv_rate);
                    $total_taxes = $precio_con_impuesto - $subtotal;
                } else {
                    $subtotal = ($price * $quantity) - $discount;
                    $total_taxes = $subtotal * $igv_rate;
                }
                $total_item = $subtotal + $total_taxes;
                
                // Guardar precio unitario real (igual que tm_detalle_venta.precio / imp_venta_all) para no alterar resultado
                $stmt_item = $this->db->prepare($sql_item);
                $stmt_item->execute(array(
                    $id_documento_fe,
                    isset($item['id_producto']) ? $item['id_producto'] : null,
                    $line,
                    $price, // Precio unitario real del producto (sin descontar descuento), igual que ticket
                    'und', // mu: unidad de medida
                    $quantity,
                    isset($item['descripcion']) ? $item['descripcion'] : 'Producto',
                    isset($item['codigo']) ? $item['codigo'] : null,
                    $discount,
                    $subtotal,
                    $total_taxes,
                    $total_item
                ));
                
                $id_item_fe = $this->db->lastInsertId();
                
                // Guardar impuestos del item
                $this->guardar_item_taxes_fe($id_item_fe, $subtotal, $igv_rate);
                
                $line++;
            }
        } catch (Exception $e) {
            error_log("Error al guardar items FE: " . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Mapear tasa de impuesto a código según catálogo DGI de Panamá
     * Códigos comunes: 00=Exento, 01=7%, 02=10%, 03=15%
     */
    private function mapear_tasa_a_codigo($rate)
    {
        $rate_percent = round($rate * 100, 2);
        
        // Mapear según tasas comunes de ITBMS en Panamá
        if ($rate_percent == 0 || abs($rate_percent) < 0.01) {
            return '00'; // Exento
        } elseif (abs($rate_percent - 7) < 0.01) {
            return '01'; // 7%
        } elseif (abs($rate_percent - 10) < 0.01) {
            return '02'; // 10%
        } elseif (abs($rate_percent - 15) < 0.01) {
            return '03'; // 15%
        } else {
            // Si no coincide con ninguna tasa estándar, usar '01' por defecto (7%)
            error_log("guardar_item_taxes_fe: Tasa no estándar detectada: " . $rate_percent . "%, usando código '01' por defecto");
            return '01';
        }
    }
    
    /**
     * Guardar impuestos de un item
     */
    private function guardar_item_taxes_fe($id_item_fe, $subtotal, $rate)
    {
        try {
            $amount = $subtotal * $rate;
            
            // Mapear tasa a código según catálogo DGI
            $tax_code = $this->mapear_tasa_a_codigo($rate);
            
            $sql_tax = "INSERT INTO tm_facturacion_electronica_item_taxes (
                id_item_fe, type, code, amount, rate
            ) VALUES (?, ?, ?, ?, ?)";
            
            $stmt_tax = $this->db->prepare($sql_tax);
            $stmt_tax->execute(array(
                $id_item_fe,
                '01', // type: 01=ITBMS
                $tax_code, // code: mapeado según tasa (00=Exento, 01=7%, 02=10%, 03=15%)
                $amount,
                $rate
            ));
        } catch (Exception $e) {
            error_log("Error al guardar taxes FE: " . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Guardar formas de pago del documento
     */
    private function guardar_payments_fe($id_documento_fe, $payments)
    {
        try {
            if (empty($payments) || !is_array($payments)) {
                error_log("guardar_payments_fe: No hay pagos para guardar o no es un array. Payments recibido: " . print_r($payments, true));
                return;
            }
            
            error_log("guardar_payments_fe: Guardando " . count($payments) . " pagos para documento FE ID: " . $id_documento_fe);
            
            foreach ($payments as $index => $payment) {
                if (empty($payment) || !is_array($payment)) {
                    error_log("guardar_payments_fe: Pago en índice $index está vacío o no es un array: " . print_r($payment, true));
                    continue;
                }
                
                $sql_payment = "INSERT INTO tm_facturacion_electronica_payments (
                    id_documento_fe, id_tipo_pago, type, amount, description
                ) VALUES (?, ?, ?, ?, ?)";
                
                // Mapear tipo de pago del sistema al tipo de facturafacil
                $payment_type = isset($payment['tipo']) ? $this->mapear_tipo_pago($payment['tipo']) : '99';
                $amount = isset($payment['monto']) ? floatval($payment['monto']) : 0;
                $description = isset($payment['descripcion']) ? $payment['descripcion'] : null;
                $id_tipo_pago = isset($payment['id_tipo_pago']) ? $payment['id_tipo_pago'] : null;
                
                // Validar que el monto sea mayor a 0
                if ($amount <= 0) {
                    error_log("guardar_payments_fe: Pago en índice $index tiene monto 0 o negativo, se omite");
                    continue;
                }
                
                error_log("guardar_payments_fe: Insertando pago - Tipo: $payment_type, Monto: $amount, Descripción: $description");
                
                $stmt_payment = $this->db->prepare($sql_payment);
                $result = $stmt_payment->execute(array(
                    $id_documento_fe,
                    $id_tipo_pago,
                    $payment_type,
                    $amount,
                    $description
                ));
                
                if ($result) {
                    $id_payment = $this->db->lastInsertId();
                    error_log("guardar_payments_fe: Pago guardado exitosamente con ID: $id_payment");
                } else {
                    $error_info = $stmt_payment->errorInfo();
                    error_log("guardar_payments_fe: Error al ejecutar INSERT - " . print_r($error_info, true));
                }
            }
        } catch (Exception $e) {
            error_log("Error al guardar payments FE: " . $e->getMessage());
            error_log("Stack trace: " . $e->getTraceAsString());
            // No lanzar la excepción para que no detenga el proceso completo
        }
    }
    
    /**
     * Mapear tipo de pago del sistema al tipo de facturafacil
     */
    private function mapear_tipo_pago($tipo_pago_sistema)
    {
        // Mapeo básico: ajustar según los tipos de pago del sistema
        $mapeo = array(
            '1' => '02', // Efectivo
            '2' => '03', // Tarjeta de crédito
            '3' => '04', // Tarjeta de débito
            '4' => '08', // Transferencia
            '5' => '09', // Cheque
            '6' => '10', // POS
            '7' => '01', // Crédito
            '8' => '99'  // Otro
        );
        
        return isset($mapeo[$tipo_pago_sistema]) ? $mapeo[$tipo_pago_sistema] : '99';
    }
    
    /**
     * Actualizar correlativo del tipo de documento después de emitir (tm_tipo_doc)
     */
    private function actualizar_correlativo_tipodoc($id_tipo_doc, $fd_number_usado)
    {
        try {
            // Obtener el tipo de documento actual
            $stm_tipodoc = $this->db->prepare("SELECT numero_actual, numero_final, numero FROM tm_tipo_doc WHERE id_tipo_doc = ?");
            $stm_tipodoc->execute(array($id_tipo_doc));
            $tipodoc = $stm_tipodoc->fetch(PDO::FETCH_ASSOC);
            
            if ($tipodoc) {
                // Si se usó un número específico, usar ese; si no, incrementar el actual
                if ($fd_number_usado > 0) {
                    $nuevo_correlativo = $fd_number_usado + 1;
                } else {
                    $numero_actual = $tipodoc['numero_actual'] ?: ($tipodoc['numero'] ? intval($tipodoc['numero']) : 1);
                    $nuevo_correlativo = intval($numero_actual) + 1;
                }
                
                // Verificar que no exceda el número final
                $numero_final = $tipodoc['numero_final'] ?: 9999999999;
                if ($nuevo_correlativo > intval($numero_final)) {
                    error_log("actualizar_correlativo_tipodoc: El correlativo $nuevo_correlativo excede el número final " . $numero_final);
                    return false;
                }
                
                // Actualizar el correlativo en tm_tipo_doc (numero_actual y numero para compatibilidad)
                $stm_update = $this->db->prepare("UPDATE tm_tipo_doc SET numero_actual = ?, numero = ? WHERE id_tipo_doc = ?");
                $result = $stm_update->execute(array($nuevo_correlativo, str_pad($nuevo_correlativo, 8, '0', STR_PAD_LEFT), $id_tipo_doc));
                
                if ($result) {
                    error_log("actualizar_correlativo_tipodoc: Correlativo actualizado exitosamente. Tipo Doc ID: $id_tipo_doc, Nuevo correlativo: $nuevo_correlativo");
                    return true;
                } else {
                    $error_info = $stm_update->errorInfo();
                    error_log("actualizar_correlativo_tipodoc: Error al actualizar correlativo - " . print_r($error_info, true));
                    return false;
                }
            } else {
                error_log("actualizar_correlativo_tipodoc: No se encontró el tipo de documento con ID: $id_tipo_doc");
                return false;
            }
        } catch (Exception $e) {
            error_log("actualizar_correlativo_tipodoc: Excepción - " . $e->getMessage());
            return false;
        }
    }

    /**
     * Actualizar correlativo de la serie de documento después de emitir
     */
    private function actualizar_correlativo_serie($id_serie, $fd_number_usado)
    {
        try {
            // Obtener la serie actual
            $stm_serie = $this->db->prepare("SELECT numero_actual, numero_final FROM tm_series_documentos WHERE id_serie = ?");
            $stm_serie->execute(array($id_serie));
            $serie = $stm_serie->fetch(PDO::FETCH_ASSOC);
            
            if ($serie) {
                // Si se usó un número específico, usar ese; si no, incrementar el actual
                if ($fd_number_usado > 0) {
                    $nuevo_correlativo = $fd_number_usado + 1;
                } else {
                    $nuevo_correlativo = intval($serie['numero_actual']) + 1;
                }
                
                // Verificar que no exceda el número final
                if ($nuevo_correlativo > intval($serie['numero_final'])) {
                    error_log("actualizar_correlativo_serie: El correlativo $nuevo_correlativo excede el número final " . $serie['numero_final']);
                    return false;
                }
                
                // Actualizar el correlativo
                $stm_update = $this->db->prepare("UPDATE tm_series_documentos SET numero_actual = ? WHERE id_serie = ?");
                $result = $stm_update->execute(array($nuevo_correlativo, $id_serie));
                
                if ($result) {
                    error_log("actualizar_correlativo_serie: Correlativo actualizado exitosamente. Serie ID: $id_serie, Nuevo correlativo: $nuevo_correlativo");
                    return true;
                } else {
                    $error_info = $stm_update->errorInfo();
                    error_log("actualizar_correlativo_serie: Error al actualizar correlativo - " . print_r($error_info, true));
                    return false;
                }
            } else {
                error_log("actualizar_correlativo_serie: No se encontró la serie con ID: $id_serie");
                return false;
            }
        } catch (Exception $e) {
            error_log("actualizar_correlativo_serie: Excepción al actualizar correlativo - " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Listar documentos para DataTable
     */
    public function documentos_list($data)
    {
        try {
            $start = isset($data['start']) ? intval($data['start']) : 0;
            $length = isset($data['length']) ? intval($data['length']) : 10;
            $draw = isset($data['draw']) ? intval($data['draw']) : 1;
            
            $where = "WHERE 1=1";
            $params = array();
            
            // Filtros - usar alias 'd.' desde el inicio para evitar ambigüedad con JOIN
            if (isset($data['fecha_desde']) && !empty($data['fecha_desde'])) {
                $where .= " AND DATE(d.fecha_creacion) >= ?";
                $params[] = $data['fecha_desde'];
            }
            
            if (isset($data['fecha_hasta']) && !empty($data['fecha_hasta'])) {
                $where .= " AND DATE(d.fecha_creacion) <= ?";
                $params[] = $data['fecha_hasta'];
            }
            
            if (isset($data['status']) && !empty($data['status'])) {
                $where .= " AND d.status = ?";
                $params[] = $data['status'];
            }
            
            if (isset($data['tipo_documento']) && !empty($data['tipo_documento'])) {
                $where .= " AND d.document_type = ?";
                $params[] = $data['tipo_documento'];
            }
            
            // Contar total de registros (usar alias 'd' para consistencia)
            $sql_count = "SELECT COUNT(*) as total FROM tm_facturacion_electronica_documentos d $where";
            $stm_count = $this->db->prepare($sql_count);
            $stm_count->execute($params);
            $total_records = $stm_count->fetch(PDO::FETCH_ASSOC)['total'];
            
            // Obtener registros con paginación
            // Buscar serie desde tm_series_documentos O desde tm_tipo_doc (si id_serie_documento es NULL)
            $sql = "SELECT d.*, 
                    COALESCE(s.serie, td.serie) as serie_electronica,
                    v.id_tipo_doc as id_tipo_doc_venta,
                    td.descripcion as tipo_doc_descripcion
                    FROM tm_facturacion_electronica_documentos d
                    LEFT JOIN tm_series_documentos s ON d.id_serie_documento = s.id_serie
                    LEFT JOIN tm_venta v ON d.id_venta = v.id_venta
                    LEFT JOIN tm_tipo_doc td ON v.id_tipo_doc = td.id_tipo_doc AND d.id_serie_documento IS NULL
                    $where 
                    ORDER BY d.id_documento_fe DESC 
                    LIMIT $start, $length";
            
            // Log para debug
            error_log("documentos_list SQL: " . $sql);
            error_log("documentos_list Params: " . print_r($params, true));
            error_log("documentos_list Start: $start, Length: $length");
            
            $stm = $this->db->prepare($sql);
            $stm->execute($params);
            $records = $stm->fetchAll(PDO::FETCH_ASSOC);
            
            error_log("documentos_list Records encontrados: " . count($records));
            
            // Serie/pos correcta para NTC/NTD (tm_tipo_doc 4=NTC, 5=NTD)
            $stm_tipo_nt = $this->db->query("SELECT id_tipo_doc, serie, pos FROM tm_tipo_doc WHERE id_tipo_doc IN (4, 5)");
            $tipos_nt = array();
            while ($row = $stm_tipo_nt->fetch(PDO::FETCH_ASSOC)) {
                $tipos_nt[$row['id_tipo_doc']] = $row;
            }
            // Para cada FAC: ver si existe NTC que lo referencie (referred_fd_number = cufe o fd_number)
            $stm_ntc_ref = $this->db->query("SELECT n.referred_fd_number, n.fd_number, n.document_type,
                (SELECT serie FROM tm_tipo_doc WHERE id_tipo_doc = IF(n.document_type='NTC', 4, 5) LIMIT 1) as serie_ntc
                FROM tm_facturacion_electronica_documentos n
                WHERE n.document_type IN ('NTC', 'NTD') AND n.referred_fd_number IS NOT NULL AND TRIM(n.referred_fd_number) != ''");
            $ntc_por_referencia = array();
            while ($row = $stm_ntc_ref->fetch(PDO::FETCH_ASSOC)) {
                $ref = trim($row['referred_fd_number']);
                $ntc_por_referencia[$ref] = $row['serie_ntc'] . '-' . str_pad($row['fd_number'], 10, '0', STR_PAD_LEFT);
            }
            // Asegurar que los valores NULL se manejen correctamente y corregir serie para NTC/NTD
            foreach ($records as &$record) {
                $record['serie_electronica'] = $record['serie_electronica'] ?? '';
                $record['receptor_name'] = $record['receptor_name'] ?? 'N/A';
                $record['total'] = $record['total'] ?? 0;
                $record['status'] = $record['status'] ?? 'pending';
                $record['cufe'] = $record['cufe'] ?? null;
                if (isset($record['document_type']) && $record['document_type'] == 'NTC' && isset($tipos_nt[4])) {
                    $record['serie_electronica'] = $tipos_nt[4]['serie'] ?: 'NC01';
                    $record['pos'] = $tipos_nt[4]['pos'] ?: '001';
                } elseif (isset($record['document_type']) && $record['document_type'] == 'NTD' && isset($tipos_nt[5])) {
                    $record['serie_electronica'] = $tipos_nt[5]['serie'] ?: 'ND01';
                    $record['pos'] = $tipos_nt[5]['pos'] ?: '001';
                }
                // Para FAC: indicar si tiene NTC emitida (nota de crédito emitida)
                $record['ntc_emitida'] = '';
                if (isset($record['document_type']) && $record['document_type'] == 'FAC') {
                    $ref_cufe = $record['cufe'] ? trim($record['cufe']) : '';
                    $ref_fd = $record['fd_number'] ? str_pad($record['fd_number'], 10, '0', STR_PAD_LEFT) : '';
                    if ($ref_cufe && isset($ntc_por_referencia[$ref_cufe])) {
                        $record['ntc_emitida'] = $ntc_por_referencia[$ref_cufe];
                    } elseif ($ref_fd && isset($ntc_por_referencia[$ref_fd])) {
                        $record['ntc_emitida'] = $ntc_por_referencia[$ref_fd];
                    }
                }
            }
            unset($record);
            
            // Retornar array para que el controlador envíe el JSON
            return array(
                "draw" => intval($draw),
                "recordsTotal" => intval($total_records),
                "recordsFiltered" => intval($total_records),
                "data" => $records
            );
            
        } catch (Exception $e) {
            error_log("Error en documentos_list: " . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Obtener datos de un documento específico
     */
    public function documentos_data($data)
    {
        try {
            $id = isset($data['id_documento_fe']) ? intval($data['id_documento_fe']) : 0;
            
            $sql = "SELECT d.*, 
                    COALESCE(s.serie, td.serie) as serie_electronica,
                    v.id_cliente as id_cliente,
                    v.id_tipo_doc as id_tipo_doc_venta
                    FROM tm_facturacion_electronica_documentos d
                    LEFT JOIN tm_series_documentos s ON d.id_serie_documento = s.id_serie
                    LEFT JOIN tm_venta v ON d.id_venta = v.id_venta
                    LEFT JOIN tm_tipo_doc td ON v.id_tipo_doc = td.id_tipo_doc AND d.id_serie_documento IS NULL
                    WHERE d.id_documento_fe = ?";
            
            $stm = $this->db->prepare($sql);
            $stm->execute(array($id));
            $documento = $stm->fetch(PDO::FETCH_ASSOC);
            
            if ($documento) {
                // Para NTC/NTD usar serie y pos de tm_tipo_doc (4=NTC, 5=NTD), no de la venta
                if (isset($documento['document_type']) && in_array($documento['document_type'], array('NTC', 'NTD'))) {
                    $id_tipo_nt = ($documento['document_type'] == 'NTC') ? 4 : 5;
                    $stm_tipo = $this->db->prepare("SELECT serie, pos FROM tm_tipo_doc WHERE id_tipo_doc = ? LIMIT 1");
                    $stm_tipo->execute(array($id_tipo_nt));
                    $tipo_nt = $stm_tipo->fetch(PDO::FETCH_ASSOC);
                    if ($tipo_nt) {
                        $documento['serie_electronica'] = $tipo_nt['serie'] ?: ($documento['document_type'] == 'NTC' ? 'NC01' : 'ND01');
                        $documento['pos'] = $tipo_nt['pos'] ?: '001';
                    }
                }
                // Obtener items
                $sql_items = "SELECT i.*, t.type as tax_type, t.code as tax_code, t.amount as tax_amount
                              FROM tm_facturacion_electronica_items i
                              LEFT JOIN tm_facturacion_electronica_item_taxes t ON i.id_item_fe = t.id_item_fe
                              WHERE i.id_documento_fe = ?
                              ORDER BY i.line";
                $stm_items = $this->db->prepare($sql_items);
                $stm_items->execute(array($id));
                $documento['items'] = $stm_items->fetchAll(PDO::FETCH_ASSOC);
                
                // Obtener pagos
                $sql_payments = "SELECT * FROM tm_facturacion_electronica_payments WHERE id_documento_fe = ?";
                $stm_payments = $this->db->prepare($sql_payments);
                $stm_payments->execute(array($id));
                $documento['payments'] = $stm_payments->fetchAll(PDO::FETCH_ASSOC);
                
                return array('success' => true, 'data' => $documento);
            } else {
                return array('success' => false, 'message' => 'Documento no encontrado');
            }
            
        } catch (Exception $e) {
            error_log("Error en documentos_data: " . $e->getMessage());
            return array('success' => false, 'message' => $e->getMessage());
        }
    }
    
    /**
     * Verificar si el envío automático está activado (tm_facturacion_electronica.envio_automatico = 1)
     */
    public function envio_automatico_activo()
    {
        try {
            $stm = $this->db->prepare("SELECT envio_automatico FROM tm_facturacion_electronica WHERE id_facturacion = 1");
            $stm->execute();
            $row = $stm->fetch(PDO::FETCH_ASSOC);
            return $row && isset($row['envio_automatico']) && (int)$row['envio_automatico'] === 1;
        } catch (Exception $e) {
            return false;
        }
    }
    
    /**
     * Obtener IDs de documentos pendientes de envío (status = 'pending', NULL o vacío).
     * Coincide con lo que la vista facturacion/documentos muestra como "Pendiente".
     */
    public function get_documentos_pendientes_envio($limite = 50)
    {
        try {
            $limite = max(1, min(100, (int)$limite));
            $sql = "SELECT id_documento_fe FROM tm_facturacion_electronica_documentos 
                    WHERE (status = 'pending' OR status IS NULL OR TRIM(COALESCE(status,'')) = '')
                    ORDER BY id_documento_fe ASC LIMIT " . $limite;
            $stm = $this->db->prepare($sql);
            $stm->execute();
            $ids = array();
            while ($row = $stm->fetch(PDO::FETCH_ASSOC)) {
                $ids[] = (int)$row['id_documento_fe'];
            }
            return $ids;
        } catch (Exception $e) {
            error_log("Error en get_documentos_pendientes_envio: " . $e->getMessage());
            return array();
        }
    }
    
    /**
     * Enviar documento a facturafacil
     */
    public function enviar_documento_fe($data)
    {
        try {
            $id_documento_fe = isset($data['id_documento_fe']) ? intval($data['id_documento_fe']) : 0;
            
            // Obtener documento completo
            $documento_data = $this->documentos_data(array('id_documento_fe' => $id_documento_fe));
            
            if (!$documento_data['success']) {
                return array('success' => false, 'message' => 'Documento no encontrado');
            }
            
            $doc = $documento_data['data'];
            
            // Obtener credenciales de facturación electrónica
            $stm_fe = $this->db->prepare("SELECT * FROM tm_facturacion_electronica WHERE id_facturacion = 1");
            $stm_fe->execute();
            $credenciales = $stm_fe->fetch(PDO::FETCH_ASSOC);
            
            if (!$credenciales || $credenciales['activo'] != 1) {
                return array('success' => false, 'message' => 'La facturación electrónica no está activa');
            }
            
            // Validar que el branch esté configurado (requerido por la API)
            if (empty($credenciales['ff_branch'])) {
                return array('success' => false, 'message' => 'La sucursal (branch) no está configurada. Por favor configurela en los ajustes del sistema. El código de sucursal debe tener 4 caracteres.');
            }
            
            // Construir el JSON según la documentación Swagger
            $payload = $this->construir_payload_api($doc, $credenciales);
            
            $log_file_fe = __DIR__ . '/../debug_facturacion_electronica.log';
            $log_timbrar = date('Y-m-d H:i:s') . " [TIMBRAR] id_documento_fe=" . $doc['id_documento_fe'] . " id_venta=" . $doc['id_venta'] . "\n";
            $log_timbrar .= "  document->receptor (enviado a DGI): " . json_encode($payload['document']['receptor'], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT) . "\n";
            $log_timbrar .= "  document->receptor->location: " . (isset($payload['document']['receptor']['location']) ? "'" . $payload['document']['receptor']['location'] . "'" : 'NO PRESENTE') . "\n";
            $log_timbrar .= "  document->fd_number: " . (isset($payload['document']['fd_number']) ? $payload['document']['fd_number'] : 'N/A') . "\n";
            @file_put_contents($log_file_fe, $log_timbrar, FILE_APPEND);
            error_log("=== TIMBRAR - Payload receptor enviado a DGI ===");
            error_log("document->receptor: " . json_encode($payload['document']['receptor'], JSON_UNESCAPED_UNICODE));
            error_log("document->receptor->location: " . (isset($payload['document']['receptor']['location']) ? $payload['document']['receptor']['location'] : 'NO ENVIADO'));
            
            // Log detallado del payload para debug
            error_log("=== PAYLOAD COMPLETO ENVIADO A API ===");
            error_log("Payload completo: " . json_encode($payload, JSON_PRETTY_PRINT));
            error_log("Header específico: " . json_encode($payload['header'], JSON_PRETTY_PRINT));
            error_log("¿Tiene header.id?: " . (isset($payload['header']['id']) ? 'SÍ - ' . $payload['header']['id'] : 'NO'));
            error_log("¿Tiene header.system_ref?: " . (isset($payload['header']['system_ref']) ? 'SÍ - ' . $payload['header']['system_ref'] : 'NO'));
            error_log("Header completo JSON: " . json_encode($payload['header']));
            error_log("=== FIN PAYLOAD ===");
            
            // Determinar URL según ambiente
            $url_api = ($doc['environment'] == 1) 
                ? 'https://backend-api.facturafacil.com.pa/api/pac/reception_fe/detailed/'
                : 'https://backend-qa-api.facturafacil.com.pa/api/pac/reception_fe/detailed/';
            
            // Enviar a la API
            $ch = curl_init();
            curl_setopt_array($ch, array(
                CURLOPT_URL => $url_api,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_ENCODING => '',
                CURLOPT_MAXREDIRS => 10,
                CURLOPT_TIMEOUT => 30,
                CURLOPT_FOLLOWLOCATION => true,
                CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
                CURLOPT_CUSTOMREQUEST => 'POST',
                CURLOPT_POSTFIELDS => json_encode($payload),
                CURLOPT_HTTPHEADER => array(
                    'X-FF-Company: ' . $credenciales['ff_company_uuid'],
                    'X-FF-API-Key: ' . $credenciales['ff_api_key'],
                    'X-FF-Branch: ' . ($credenciales['ff_branch'] ?: ''),
                    'Content-Type: application/json',
                    'Accept: application/json'
                ),
            ));
            
            // Log del JSON que se envía exactamente
            $json_payload = json_encode($payload);
            error_log("=== JSON ENVIADO A API ===");
            error_log("JSON completo: " . $json_payload);
            error_log("Tamaño del JSON: " . strlen($json_payload) . " bytes");
            error_log("Header en JSON: " . json_encode($payload['header']));
            error_log("¿Tiene header.id?: " . (isset($payload['header']['id']) ? 'SÍ - ' . $payload['header']['id'] : 'NO'));
            error_log("¿Tiene header.system_ref?: " . (isset($payload['header']['system_ref']) ? 'SÍ - ' . $payload['header']['system_ref'] : 'NO'));
            error_log("=== FIN JSON ===");
            
            $response = curl_exec($ch);
            $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $curl_error = curl_error($ch);
            
            // Log de la respuesta
            error_log("=== RESPUESTA DE API ===");
            error_log("HTTP Code: " . $http_code);
            error_log("Response: " . $response);
            error_log("CURL Error: " . ($curl_error ?: 'Ninguno'));
            error_log("=== FIN RESPUESTA ===");
            
            curl_close($ch);
            
            if ($curl_error) {
                error_log("Error CURL al enviar documento FE: " . $curl_error);
                return array('success' => false, 'message' => 'Error de conexión: ' . $curl_error);
            }
            
            $response_data = json_decode($response, true);
            
            // Log completo de la respuesta para verificar qué datos devuelve la API
            error_log("=== RESPUESTA COMPLETA DE LA API ===");
            error_log("HTTP Code: " . $http_code);
            error_log("Response completa: " . json_encode($response_data, JSON_PRETTY_PRINT));
            error_log("Campos en response_data: " . implode(', ', array_keys($response_data ?: array())));
            error_log("=== FIN RESPUESTA API ===");
            
            // Actualizar documento con la respuesta
            $this->actualizar_documento_respuesta($id_documento_fe, $response_data, $http_code);
            
            if ($http_code == 201) {
                if (isset($response_data['rejected']) && $response_data['rejected'] == true) {
                    return array(
                        'success' => true, 
                        'message' => 'Documento enviado pero rechazado por DGI',
                        'rejected' => true,
                        'messages' => isset($response_data['messages']) ? $response_data['messages'] : array()
                    );
                } else {
                    return array(
                        'success' => true, 
                        'message' => 'Documento enviado y autorizado correctamente',
                        'rejected' => false,
                        'cufe' => isset($response_data['cufe']) ? $response_data['cufe'] : null
                    );
                }
            } else {
                $error_msg = isset($response_data['messages']) && is_array($response_data['messages']) 
                    ? implode(', ', array_column($response_data['messages'], 'message'))
                    : 'Error al enviar el documento';
                
                return array('success' => false, 'message' => $error_msg);
            }
            
        } catch (Exception $e) {
            error_log("Error en enviar_documento_fe: " . $e->getMessage());
            return array('success' => false, 'message' => $e->getMessage());
        }
    }
    
    /**
     * Construir payload para la API según documentación Swagger
     */
    private function construir_payload_api($doc, $credenciales)
    {
        // Obtener items
        $items = array();
        $sql_items = "SELECT * FROM tm_facturacion_electronica_items WHERE id_documento_fe = ? ORDER BY line";
        $stm_items = $this->db->prepare($sql_items);
        $stm_items->execute(array($doc['id_documento_fe']));
        $items_db = $stm_items->fetchAll(PDO::FETCH_ASSOC);
        
        $total_items_calculado = 0;
        foreach ($items_db as $item) {
            // Obtener impuestos de este item
            $taxes = array();
            $sql_taxes = "SELECT * FROM tm_facturacion_electronica_item_taxes WHERE id_item_fe = ?";
            $stm_taxes = $this->db->prepare($sql_taxes);
            $stm_taxes->execute(array($item['id_item_fe']));
            $taxes_db = $stm_taxes->fetchAll(PDO::FETCH_ASSOC);
            
            // Swagger/DGI: subtotal = price * quantity - discount; redondeo 2 decimales (half-up).
            // Usar subtotal BASE (sin impuesto) para que el monto ITBMS sea válido (evitar error 2152).
            $quantity_item = floatval($item['quantity']);
            $discount_item = floatval($item['discount']);
            if ($quantity_item <= 0) {
                $quantity_item = 1;
            }
            $subtotal_bd = isset($item['subtotal']) ? floatval($item['subtotal']) : null;
            if ($subtotal_bd !== null && $subtotal_bd > 0) {
                $subtotal_item = round($subtotal_bd, 2);
                // Precio unitario tal que price*quantity - discount = subtotal (Swagger permite hasta 6 decimales)
                $price_unit = ($subtotal_item + $discount_item) / $quantity_item;
                $price_unit = round($price_unit, 6);
            } else {
                // Fallback: calcular desde price/quantity/discount de BD
                $price_unit = floatval($item['price']);
                $subtotal_item = round($price_unit * $quantity_item - $discount_item, 2);
                $price_unit = round($price_unit, 6);
            }
            
            $total_taxes_item = 0;
            foreach ($taxes_db as $tax) {
                $tax_rate = isset($tax['rate']) ? floatval($tax['rate']) : 0.07;
                // Recalcular monto ITBMS como round(subtotal * tasa, 2) para que DGI no rechace (error 2152)
                $tax_amount = round($subtotal_item * $tax_rate, 2);
                $total_taxes_item += $tax_amount;
                
                $tax_data = array(
                    'type' => $tax['type'] ?: '01',
                    'amount' => number_format($tax_amount, 2, '.', '')
                );
                if ($tax['code']) {
                    $tax_data['code'] = $tax['code'];
                } else {
                    $tax_data['code'] = $this->mapear_tasa_a_codigo($tax_rate);
                }
                if ($tax['rate']) {
                    $tax_data['rate'] = number_format(floatval($tax['rate']) * 100, 2, '.', '');
                }
                $taxes[] = $tax_data;
            }
            $total_taxes_item = round($total_taxes_item, 2);
            
            // Total ítem = subtotal + impuestos (redondeado 2 decimales para coincidir con dVTotItems y evitar error 2514)
            $total_item = round($subtotal_item + $total_taxes_item, 2);
            
            // Log para verificar que se están usando los valores correctos
            error_log("construir_payload_api: Item línea " . $item['line'] . " - Precio enviado: " . $price_unit . ", Subtotal: " . $subtotal_item . ", ITBMS: " . $total_taxes_item . ", Total ítem: " . $total_item);
            
            // Swagger: price hasta 6 decimales; quantity, discount como string decimal 2 decimales
            $item_data = array(
                'line' => intval($item['line']),
                'price' => number_format($price_unit, 6, '.', ''),
                'mu' => $item['mu'] ?: 'und',
                'quantity' => number_format(floatval($item['quantity']), 2, '.', ''),
                'description' => $item['description'],
                'internal_code' => $item['internal_code'] ?: '',
                'discount' => number_format($discount_item, 2, '.', ''),
                'taxes' => $taxes
            );
            
            // Código CPBS/GNS - REQUERIDO por DGI (error 2007 si no se envía). Swagger DocumentItem: campo "gns", maxLength 4
            $cpbs_code = '';
            if (isset($item['gns']) && trim($item['gns']) !== '') {
                $cpbs_code = substr(trim($item['gns']), 0, 4);
                if (strlen($cpbs_code) == 4 && ctype_digit($cpbs_code)) {
                    $item_data['gns'] = $cpbs_code;
                    error_log("construir_payload_api: Item línea " . $item['line'] . " gns desde BD: " . $cpbs_code);
                } else {
                    $item_data['gns'] = '5010';
                    error_log("construir_payload_api: Item línea " . $item['line'] . " gns inválido, usando 5010");
                }
            } else {
                $item_data['gns'] = '5010'; // Servicios de restaurante (UNSPSC)
                error_log("construir_payload_api: Item línea " . $item['line'] . " sin gns, usando 5010");
            }
            
            $items[] = $item_data;
            $total_items_calculado += $total_item;
        }
        
        // Redondear suma total a 2 decimales (debe coincidir con total y dVTotItems para evitar error 2514)
        $total_items_calculado = round($total_items_calculado, 2);
        
        error_log("construir_payload_api: ===== CÁLCULO dVTotItems =====");
        error_log("construir_payload_api: Total items calculado (suma manual): " . $total_items_calculado);
        error_log("construir_payload_api: Total items formateado: " . number_format($total_items_calculado, 2, '.', ''));
        error_log("construir_payload_api: Total del documento: " . $doc['total']);
        error_log("construir_payload_api: ===== FIN CÁLCULO =====");
        
        // Obtener pagos
        $payments = array();
        $sql_payments = "SELECT * FROM tm_facturacion_electronica_payments WHERE id_documento_fe = ?";
        $stm_payments = $this->db->prepare($sql_payments);
        $stm_payments->execute(array($doc['id_documento_fe']));
        $payments_db = $stm_payments->fetchAll(PDO::FETCH_ASSOC);
        
        foreach ($payments_db as $payment) {
            // Asegurar que la descripción tenga al menos 10 caracteres
            $description = trim($payment['description'] ?: '');
            if (strlen($description) < 10) {
                // Generar descripción basada en el tipo de pago si es muy corta
                $description = $this->generar_descripcion_pago($payment['type'], $description);
            }
            
            $payments[] = array(
                'type' => $payment['type'],
                'amount' => round(floatval($payment['amount']), 2),
                'description' => $description
            );
        }
        
        // Si no hay pagos, crear uno por defecto (requerido por la API). amount como number (Swagger DocumentPayment)
        if (empty($payments)) {
            $total_doc = floatval($doc['total']);
            $payments[] = array(
                'type' => '02',
                'amount' => round($total_doc, 2),
                'description' => $this->generar_descripcion_pago('02', '')
            );
            error_log("construir_payload_api: No se encontraron pagos, se creó uno por defecto de tipo efectivo por monto: " . $total_doc);
        }
        
        // Construir receptor
        // IMPORTANTE: Si es Consumidor Final (type="02"), NO debe incluir RUC/DV/ubicación según documentación
        $receptor = array(
            'type' => $doc['receptor_type'],
            'name' => $doc['receptor_name']
        );
        
        // Contribuyente (01) o Gobierno (03): la API DGI requiere ruc, dv, address, location (Swagger)
        if ($doc['receptor_type'] != '02') {
            $receptor['ruc'] = isset($doc['receptor_ruc']) && trim($doc['receptor_ruc']) !== '' ? trim($doc['receptor_ruc']) : '';
            $receptor['ruc_type'] = isset($doc['receptor_ruc_type']) && $doc['receptor_ruc_type'] !== '' ? strval($doc['receptor_ruc_type']) : '2';
            $receptor['address'] = isset($doc['receptor_address']) && trim($doc['receptor_address']) !== '' ? substr(trim($doc['receptor_address']), 0, 100) : '';
            $receptor['dv'] = isset($doc['receptor_dv']) && trim($doc['receptor_dv']) !== '' ? substr(trim($doc['receptor_dv']), 0, 2) : '';
            // Ubicación: Swagger DocumentReceptor.location = string maxLength 8, minLength 5.
            // Descripción endpoint: "location (PP-DD-CC)" y "debe corresponder a un código en la tabla de ubicaciones".
            // Se envía en formato provincia-distrito-corregimiento (08-01-25). Si el PAC devuelve "Código de ubicación no encontrado",
            // puede que la tabla del PAC use clave de 6 dígitos: probamos enviar solo dígitos (080125) para lookup en catálogo.
            $receptor_location_raw = isset($doc['receptor_location']) && trim($doc['receptor_location']) !== '' ? trim($doc['receptor_location']) : '';
            $solo_digitos = preg_replace('/[^0-9]/', '', $receptor_location_raw);
            $solo_digitos = substr(str_pad($solo_digitos, 6, '0', STR_PAD_LEFT), 0, 6);
            $con_guiones = strlen($solo_digitos) >= 6 ? (substr($solo_digitos, 0, 2) . '-' . substr($solo_digitos, 2, 2) . '-' . substr($solo_digitos, 4, 2)) : $receptor_location_raw;
            // Swagger: location maxLength 8, minLength 5; endpoint "provincia-distrito-corregimiento". Error "Código de ubicación no encontrado"
            // suele indicar que el código no está en la tabla del PAC; algunos PAC hacen lookup por 6 dígitos. Enviamos 6 dígitos.
            // Si la API devuelve "Ubicación inválida, debe tener el formato: provincia-distrito-corregimiento", usar: $receptor['location'] = $con_guiones;
            $receptor['location'] = $solo_digitos;
            if (strlen($receptor['ruc']) < 5) {
                throw new Exception('Para Contribuyente o Gobierno el RUC del receptor es obligatorio. Edite el cliente y complete el RUC.');
            }
            if (strlen($receptor['address']) < 5) {
                throw new Exception('Para Contribuyente o Gobierno la dirección del receptor es obligatoria (mín. 5 caracteres). Edite el cliente.');
            }
            $location_ok = (preg_match('/^\d{2}-\d{2}-\d{2}$/', $receptor_location_raw) || (strlen(preg_replace('/[^0-9]/', '', $receptor_location_raw)) >= 6));
            if (strlen($receptor_location_raw) < 5 || !$location_ok) {
                throw new Exception('Para Contribuyente o Gobierno la ubicación del receptor es obligatoria (formato PP-DD-CC, ej: 08-01-01). Edite el cliente.');
            }
            error_log("construir_payload_api: receptor location raw=" . $receptor_location_raw . " enviado (6 dígitos)=" . $receptor['location']);
        }
        
        // Email se puede agregar siempre (opcional)
        if ($doc['receptor_email']) {
            $receptor['email'] = $doc['receptor_email'];
        }
        
        // Construir payload completo
        // Obtener branch de las credenciales (requerido por la API)
        // El branch debe ser un objeto DocumentBranch con al menos el campo 'code'
        $branch_code = isset($credenciales['ff_branch']) && !empty($credenciales['ff_branch']) 
            ? trim($credenciales['ff_branch']) 
            : '';
        
        // Construir header según documentación de facturafacil
        // El header debe tener "id" según el ejemplo de la documentación
        $header = array(
            'environment' => strval($doc['environment'])
        );
        
        // Agregar id al header - usar header_id si existe, sino usar id_documento_fe
        if (isset($doc['header_id']) && !empty($doc['header_id']) && trim($doc['header_id']) !== '' && trim($doc['header_id']) !== '0') {
            $header['id'] = intval($doc['header_id']);
            error_log("construir_payload_api: Usando header_id de BD: " . $doc['header_id']);
        } else {
            // Usar el ID del documento como id del header (según documentación)
            $header['id'] = intval($doc['id_documento_fe']);
            error_log("construir_payload_api: Usando id_documento_fe como header.id: " . $doc['id_documento_fe']);
        }
        
        // Agregar campos adicionales si existen (opcionales según documentación)
        // POS debe ser exactamente 3 dígitos (000-999)
        if (isset($doc['pos']) && !empty($doc['pos'])) {
            $pos_raw = trim($doc['pos']);
            // Asegurar que tenga exactamente 3 dígitos, rellenando con ceros a la izquierda
            $pos_formatted = str_pad(substr(preg_replace('/[^0-9]/', '', $pos_raw), 0, 3), 3, '0', STR_PAD_LEFT);
            $header['pos'] = $pos_formatted;
        } else {
            // Si no viene POS, usar '001' por defecto
            $header['pos'] = '001';
        }
        
        // issue_date: requerido por API (formato YYYY-MM-DD H:i:s). "La fecha de emisión no puede ser mayor a 2 días desde la fecha actual"
        $issue_date = isset($doc['issue_date']) && !empty(trim($doc['issue_date'])) ? trim($doc['issue_date']) : null;
        if (!$issue_date && isset($doc['fecha_creacion']) && !empty($doc['fecha_creacion'])) {
            $issue_date = date('Y-m-d H:i:s', strtotime($doc['fecha_creacion']));
        }
        if (!$issue_date) {
            $issue_date = date('Y-m-d H:i:s');
        }
        $header['issue_date'] = $issue_date;
        
        // Log del header antes de crear el payload
        error_log("construir_payload_api: Header ANTES de crear payload: " . json_encode($header));
        
        // Validar fd_number: debe ser un entero válido (0-9999999999)
        $fd_number = intval($doc['fd_number']);
        if ($fd_number < 0 || $fd_number > 9999999999) {
            throw new Exception('El número fiscal (fd_number) debe estar entre 0 y 9999999999');
        }
        
        $payload = array(
            'header' => $header,
            'document' => array(
                'fd_number' => $fd_number, // Entero válido (0-9999999999)
                'type' => $doc['document_type'], // FAC, NTC, NTD
                'receptor' => $receptor,
                'items' => $items,
                'payments' => $payments,
                'total' => number_format($total_items_calculado, 2, '.', ''), // Usar suma de ítems para evitar error 2514 (valor total ítems inválidos)
                'dVTotItems' => number_format($total_items_calculado, 2, '.', ''),
                'dest_country' => $doc['dest_country'] ?: 'PA', // 2 caracteres
                'payment_term' => intval($doc['payment_term']), // 1=Contado, 2=Crédito, 3=Mixto
                'sale_type' => $doc['sale_type'] ?: '1' // 1=Giro negocio, 2=Activo fijo, 3=Bienes raíces, 4=Servicios
            )
        );
        
        error_log("construir_payload_api: Total documento: " . $doc['total'] . ", Total items calculado: " . $total_items_calculado);
        
        // Agregar branch al header (requerido por la API)
        // El branch debe ser un objeto DocumentBranch, no un string
        // Si la sucursal no existe en el sistema, se requiere dirección, coordenadas, teléfono y ubicación
        if (!empty($branch_code)) {
            // Asegurar que el código tenga exactamente 4 caracteres (rellenar con ceros a la izquierda si es necesario)
            $branch_code_padded = str_pad(substr($branch_code, 0, 4), 4, '0', STR_PAD_LEFT);
            
            $branch_data = array(
                'code' => $branch_code_padded
            );
            
            // Obtener datos de la SUCURSAL desde la configuración de facturación electrónica
            // Si no están configurados, intentar obtener de la empresa como fallback
            $branch_address = isset($credenciales['branch_address']) && !empty($credenciales['branch_address']) 
                ? trim($credenciales['branch_address']) 
                : null;
            $branch_phone = isset($credenciales['branch_phone']) && !empty($credenciales['branch_phone']) 
                ? trim($credenciales['branch_phone']) 
                : null;
            $branch_location = isset($credenciales['branch_location']) && !empty($credenciales['branch_location']) 
                ? trim($credenciales['branch_location']) 
                : null;
            $branch_coordinates = isset($credenciales['branch_coordinates']) && !empty($credenciales['branch_coordinates']) 
                ? trim($credenciales['branch_coordinates']) 
                : null;
            $branch_email = isset($credenciales['branch_email']) && !empty($credenciales['branch_email']) 
                ? trim($credenciales['branch_email']) 
                : null;
            
            // Si no hay datos de sucursal configurados, intentar obtener de la empresa como fallback
            if (empty($branch_address) || empty($branch_phone) || empty($branch_location) || empty($branch_coordinates)) {
                $stm_empresa = $this->db->prepare("SELECT direccion_comercial, direccion_fiscal, celular, ubigeo FROM tm_empresa LIMIT 1");
                $stm_empresa->execute();
                $empresa_data = $stm_empresa->fetch(PDO::FETCH_ASSOC);
                
                // Usar datos de empresa solo si no están configurados en facturación electrónica
                if (empty($branch_address) && $empresa_data) {
                    if (!empty($empresa_data['direccion_comercial'])) {
                        $branch_address = trim($empresa_data['direccion_comercial']);
                    } elseif (!empty($empresa_data['direccion_fiscal'])) {
                        $branch_address = trim($empresa_data['direccion_fiscal']);
                    }
                }
                
                if (empty($branch_phone) && $empresa_data && !empty($empresa_data['celular'])) {
                    $telefono = preg_replace('/[^0-9]/', '', $empresa_data['celular']);
                    if (strlen($telefono) >= 8) {
                        $branch_phone = substr($telefono, -8);
                    } else {
                        $branch_phone = str_pad($telefono, 8, '0', STR_PAD_LEFT);
                    }
                }
                
                if (empty($branch_location) && $empresa_data && !empty($empresa_data['ubigeo'])) {
                    $branch_location = substr($empresa_data['ubigeo'], 0, 8);
                }
            }
            
            // Validar y agregar dirección (requerida si la sucursal no existe en el sistema)
            if (!empty($branch_address)) {
                $branch_data['address'] = substr($branch_address, 0, 100); // Máximo 100 caracteres
            } else {
                error_log("construir_payload_api: ERROR - No hay dirección de sucursal configurada.");
                throw new Exception('La dirección de la sucursal no está configurada. Por favor configure la dirección en Ajustes > Facturación Electrónica > Datos de la Sucursal.');
            }
            
            // Validar y agregar teléfono (requerido si la sucursal no existe, formato: 8 dígitos sin guiones)
            if (!empty($branch_phone)) {
                $telefono_limpio = preg_replace('/[^0-9]/', '', $branch_phone);
                if (strlen($telefono_limpio) >= 8) {
                    $branch_data['phone'] = substr($telefono_limpio, -8);
                } else {
                    $branch_data['phone'] = str_pad($telefono_limpio, 8, '0', STR_PAD_LEFT);
                }
            } else {
                error_log("construir_payload_api: ERROR - No hay teléfono de sucursal configurado.");
                throw new Exception('El teléfono de la sucursal no está configurado. Por favor configure el teléfono en Ajustes > Facturación Electrónica > Datos de la Sucursal.');
            }
            
            // Validar y agregar ubicación (requerida si la sucursal no existe, formato: PP-DD-CC para Panamá)
            if (!empty($branch_location)) {
                $branch_data['location'] = substr($branch_location, 0, 8); // Máximo 8 caracteres
            } else {
                error_log("construir_payload_api: ERROR - No hay ubicación de sucursal configurada.");
                throw new Exception('La ubicación de la sucursal no está configurada. Por favor configure la ubicación (PP-DD-CC) en Ajustes > Facturación Electrónica > Datos de la Sucursal.');
            }
            
            // Validar y agregar coordenadas (requeridas si la sucursal no existe, formato: latitud,longitud)
            if (!empty($branch_coordinates)) {
                // Validar formato de coordenadas (debe tener formato: +latitud,-longitud o latitud,longitud)
                if (preg_match('/^[+-]?\d+\.?\d*,[+-]?\d+\.?\d*$/', $branch_coordinates)) {
                    $branch_data['coordinates'] = substr($branch_coordinates, 0, 20); // Máximo 20 caracteres
                } else {
                    error_log("construir_payload_api: WARNING - Formato de coordenadas inválido: " . $branch_coordinates);
                    throw new Exception('El formato de coordenadas es inválido. Debe ser: latitud,longitud (ej: +08.9539,-79.5343)');
                }
            } else {
                error_log("construir_payload_api: ERROR - No hay coordenadas de sucursal configuradas.");
                throw new Exception('Las coordenadas de la sucursal no están configuradas. Por favor configure las coordenadas (latitud,longitud) en Ajustes > Facturación Electrónica > Datos de la Sucursal.');
            }
            
            // Agregar email si está configurado (opcional)
            if (!empty($branch_email)) {
                if (filter_var($branch_email, FILTER_VALIDATE_EMAIL)) {
                    $branch_data['email'] = substr($branch_email, 0, 100);
                } else {
                    error_log("construir_payload_api: WARNING - Email de sucursal inválido: " . $branch_email);
                }
            }
            
            // Agregar branch al header sin sobrescribir los campos existentes
            $payload['header']['branch'] = $branch_data;
        }
        
        // VERIFICACIÓN FINAL: Asegurar que siempre exista id en el header
        if (!isset($payload['header']['id']) || empty($payload['header']['id']) || $payload['header']['id'] == 0) {
            $payload['header']['id'] = intval($doc['id_documento_fe']);
            error_log("construir_payload_api: VERIFICACIÓN FINAL - Forzando id: " . $doc['id_documento_fe']);
        }
        
        // Log del header final para debug
        error_log("construir_payload_api: ===== HEADER FINAL ANTES DE ENVIAR =====");
        error_log("construir_payload_api: Header completo: " . json_encode($payload['header'], JSON_PRETTY_PRINT));
        error_log("construir_payload_api: ¿Tiene id?: " . (isset($payload['header']['id']) ? 'SÍ - ' . $payload['header']['id'] : 'NO - ERROR!'));
        error_log("construir_payload_api: Environment: " . $payload['header']['environment']);
        error_log("construir_payload_api: POS: " . (isset($payload['header']['pos']) ? $payload['header']['pos'] : 'NO DEFINIDO'));
        error_log("construir_payload_api: ===== FIN HEADER FINAL =====");
        
        // Log del documento completo antes de enviar (incl. receptor para debug ubicación DGI)
        error_log("construir_payload_api: ===== DOCUMENTO COMPLETO ANTES DE ENVIAR =====");
        error_log("construir_payload_api: fd_number: " . $payload['document']['fd_number']);
        error_log("construir_payload_api: type: " . $payload['document']['type']);
        error_log("construir_payload_api: total: " . $payload['document']['total']);
        error_log("construir_payload_api: dVTotItems: " . $payload['document']['dVTotItems']);
        error_log("construir_payload_api: receptor (completo): " . json_encode($payload['document']['receptor'], JSON_UNESCAPED_UNICODE));
        error_log("construir_payload_api: receptor.type: " . $payload['document']['receptor']['type']);
        error_log("construir_payload_api: receptor.name: " . $payload['document']['receptor']['name']);
        error_log("construir_payload_api: receptor.location (enviado a DGI): " . (isset($payload['document']['receptor']['location']) ? $payload['document']['receptor']['location'] : 'NO ENVIADO'));
        error_log("construir_payload_api: receptor.address: " . (isset($payload['document']['receptor']['address']) ? substr($payload['document']['receptor']['address'], 0, 80) . '...' : 'N/A'));
        error_log("construir_payload_api: Items count: " . count($payload['document']['items']));
        error_log("construir_payload_api: Payments count: " . count($payload['document']['payments']));
        
        // Log de cada item
        foreach ($payload['document']['items'] as $idx => $item) {
            error_log("construir_payload_api: Item " . ($idx + 1) . " - line: " . $item['line'] . ", price: " . $item['price'] . ", quantity: " . $item['quantity']);
            if (isset($item['taxes']) && is_array($item['taxes'])) {
                foreach ($item['taxes'] as $tax_idx => $tax) {
                    error_log("construir_payload_api:   Tax " . ($tax_idx + 1) . " - type: " . $tax['type'] . ", code: " . (isset($tax['code']) ? $tax['code'] : 'NO') . ", amount: " . $tax['amount'] . ", rate: " . (isset($tax['rate']) ? $tax['rate'] : 'NO'));
                }
            }
        }
        
        // Log del payload completo en formato JSON
        error_log("construir_payload_api: ===== PAYLOAD COMPLETO (JSON) =====");
        error_log(json_encode($payload, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
        error_log("construir_payload_api: ===== FIN PAYLOAD COMPLETO =====");
        
        // Agregar información adicional si existe
        if ($doc['info']) {
            $payload['document']['info'] = $doc['info'];
        }
        
        // Agregar documento referenciado si es NTC o NTD
        if (in_array($doc['document_type'], array('NTC', 'NTD'))) {
            // Intentar obtener desde campos directos
            if (isset($doc['referred_fd_number']) && $doc['referred_fd_number']) {
                $payload['document']['referred'] = array(
                    'fd_number' => $doc['referred_fd_number'],
                    'fd_date' => $doc['referred_fd_date'] ?: $doc['fecha_creacion']
                );
            } 
            // Si no, intentar obtener desde motivo_anulacion (JSON)
            else if (isset($doc['motivo_anulacion']) && !empty($doc['motivo_anulacion'])) {
                $referencia_data = json_decode($doc['motivo_anulacion'], true);
                if ($referencia_data && isset($referencia_data['fd_number'])) {
                    $payload['document']['referred'] = array(
                        'fd_number' => str_pad($referencia_data['fd_number'], 10, '0', STR_PAD_LEFT),
                        'fd_date' => $referencia_data['fd_date'] ?: $doc['fecha_creacion']
                    );
                }
            }
        }
        
        // VERIFICACIÓN FINAL ABSOLUTA antes de retornar - asegurar que id esté presente
        if (!isset($payload['header']['id']) || empty($payload['header']['id']) || $payload['header']['id'] == 0) {
            $payload['header']['id'] = intval($doc['id_documento_fe']);
            error_log("construir_payload_api: ÚLTIMA VERIFICACIÓN - id agregado antes de return: " . $doc['id_documento_fe']);
        }
        
        // Log final del payload completo antes de retornar
        error_log("construir_payload_api: ===== PAYLOAD FINAL COMPLETO =====");
        error_log("construir_payload_api: Header en payload final: " . json_encode($payload['header']));
        error_log("construir_payload_api: id presente?: " . (isset($payload['header']['id']) ? 'SÍ - ' . $payload['header']['id'] : 'NO - ERROR!'));
        error_log("construir_payload_api: ===== FIN PAYLOAD FINAL =====");
        
        return $payload;
    }
    
    /**
     * Generar descripción de pago con al menos 10 caracteres
     */
    private function generar_descripcion_pago($tipo_pago, $descripcion_original = '')
    {
        // Mapeo de tipos de pago a descripciones
        $descripciones_tipo = array(
            '01' => 'Pago a crédito',
            '02' => 'Pago en efectivo',
            '03' => 'Pago con tarjeta de crédito',
            '04' => 'Pago con tarjeta de débito',
            '05' => 'Pago con tarjeta de fidelidad',
            '06' => 'Pago con vale',
            '07' => 'Pago con tarjeta regalo',
            '08' => 'Pago por transferencia',
            '09' => 'Pago con cheque',
            '10' => 'Pago con POS',
            '99' => 'Otro método de pago'
        );
        
        // Si hay descripción original, intentar usarla con prefijo
        if (!empty($descripcion_original)) {
            $descripcion = trim($descripcion_original);
            // Si es muy corta, agregar el tipo de pago
            if (strlen($descripcion) < 10) {
                $tipo_desc = isset($descripciones_tipo[$tipo_pago]) ? $descripciones_tipo[$tipo_pago] : 'Método de pago';
                $descripcion = $tipo_desc . ' - ' . $descripcion;
            }
            // Asegurar mínimo de 10 caracteres
            while (strlen($descripcion) < 10) {
                $descripcion .= ' ';
            }
            return substr($descripcion, 0, 100); // Máximo 100 caracteres
        }
        
        // Si no hay descripción, usar la del tipo de pago
        $descripcion = isset($descripciones_tipo[$tipo_pago]) 
            ? $descripciones_tipo[$tipo_pago] 
            : 'Método de pago utilizado';
        
        // Asegurar mínimo de 10 caracteres
        while (strlen($descripcion) < 10) {
            $descripcion .= ' ';
        }
        
        return substr($descripcion, 0, 100); // Máximo 100 caracteres
    }
    
    /**
     * Actualizar documento con respuesta de la API
     */
    private function actualizar_documento_respuesta($id_documento_fe, $response_data, $http_code)
    {
        try {
            error_log("actualizar_documento_respuesta: Iniciando actualización para documento ID: " . $id_documento_fe);
            error_log("actualizar_documento_respuesta: HTTP Code: " . $http_code);
            error_log("actualizar_documento_respuesta: Response data keys: " . implode(', ', array_keys($response_data ?: array())));
            
            $update_fields = array(
                'fecha_envio' => date('Y-m-d H:i:s'),
                'status' => ($http_code == 201) ? 'sent' : 'pending'
            );
            
            // Guardar todos los campos posibles de la respuesta
            if (isset($response_data['cufe'])) {
                $update_fields['cufe'] = $response_data['cufe'];
                error_log("actualizar_documento_respuesta: Guardando CUFE: " . $response_data['cufe']);
            }
            
            if (isset($response_data['document_uuid'])) {
                $update_fields['document_uuid'] = $response_data['document_uuid'];
                error_log("actualizar_documento_respuesta: Guardando document_uuid: " . $response_data['document_uuid']);
            }
            
            if (isset($response_data['authorization_number'])) {
                $update_fields['authorization_number'] = $response_data['authorization_number'];
                error_log("actualizar_documento_respuesta: Guardando authorization_number: " . $response_data['authorization_number']);
            }
            
            if (isset($response_data['process_date'])) {
                $update_fields['process_date'] = $response_data['process_date'];
                error_log("actualizar_documento_respuesta: Guardando process_date: " . $response_data['process_date']);
            }
            
            if (isset($response_data['pdf_url'])) {
                $update_fields['pdf_url'] = $response_data['pdf_url'];
                error_log("actualizar_documento_respuesta: Guardando pdf_url: " . $response_data['pdf_url']);
            }
            
            if (isset($response_data['qr_code_data'])) {
                $update_fields['qr_code_data'] = $response_data['qr_code_data'];
                error_log("actualizar_documento_respuesta: Guardando qr_code_data");
            }
            
            if (isset($response_data['xml'])) {
                $update_fields['xml_data'] = $response_data['xml'];
                error_log("actualizar_documento_respuesta: Guardando xml_data (longitud: " . strlen($response_data['xml']) . ")");
            }
            
            if (isset($response_data['service_response'])) {
                $update_fields['service_response'] = is_array($response_data['service_response']) || is_object($response_data['service_response'])
                    ? json_encode($response_data['service_response'])
                    : $response_data['service_response'];
                error_log("actualizar_documento_respuesta: Guardando service_response");
            }
            
            if (isset($response_data['rejected'])) {
                $update_fields['rejected'] = $response_data['rejected'] ? 1 : 0;
                $update_fields['status'] = $response_data['rejected'] ? 'rejected' : 'authorized';
                error_log("actualizar_documento_respuesta: Guardando rejected: " . ($response_data['rejected'] ? '1' : '0'));
            }
            
            if (isset($response_data['messages']) && is_array($response_data['messages'])) {
                $update_fields['rejected_messages'] = json_encode($response_data['messages']);
                error_log("actualizar_documento_respuesta: Guardando rejected_messages (cantidad: " . count($response_data['messages']) . ")");
            }
            
            // Guardar cualquier otro campo que pueda venir en la respuesta
            // (por si la API devuelve campos adicionales que no conocemos)
            $campos_guardados = array_keys($update_fields);
            $campos_respuesta = array_keys($response_data ?: array());
            $campos_no_guardados = array_diff($campos_respuesta, array(
                'cufe', 'document_uuid', 'authorization_number', 'process_date', 
                'pdf_url', 'qr_code_data', 'xml', 'service_response', 
                'rejected', 'messages'
            ));
            
            if (!empty($campos_no_guardados)) {
                error_log("actualizar_documento_respuesta: ADVERTENCIA - Campos en respuesta que no se están guardando: " . implode(', ', $campos_no_guardados));
            }
            
            // Construir UPDATE
            $set_parts = array();
            $params = array();
            foreach ($update_fields as $field => $value) {
                $set_parts[] = "$field = ?";
                $params[] = $value;
            }
            $params[] = $id_documento_fe;
            
            $sql = "UPDATE tm_facturacion_electronica_documentos SET " . implode(', ', $set_parts) . " WHERE id_documento_fe = ?";
            $stm = $this->db->prepare($sql);
            $stm->execute($params);
            
        } catch (Exception $e) {
            error_log("Error al actualizar documento respuesta: " . $e->getMessage());
        }
    }
    
    /**
     * Datos para comprobante/ticket formato Panamá (documento + empresa)
     */
    public function comprobante_panama_data($id_documento_fe)
    {
        $id = intval($id_documento_fe);
        if ($id <= 0) return array('success' => false, 'message' => 'ID inválido');
        $res = $this->documentos_data(array('id_documento_fe' => $id));
        if (!$res['success']) return $res;
        $doc = $res['data'];
        $stm_empresa = $this->db->prepare("SELECT * FROM tm_empresa LIMIT 1");
        $stm_empresa->execute();
        $empresa = $stm_empresa->fetch(PDO::FETCH_ASSOC);
        if (!$empresa) $empresa = array('nombre_comercial' => NAME_NEGOCIO, 'ruc' => '', 'razon_social' => '', 'direccion_comercial' => '', 'celular' => '');
        return array('success' => true, 'doc' => $doc, 'empresa' => $empresa);
    }
    
    /**
     * Descargar PDF del documento
     */
    public function descargar_pdf($data)
    {
        try {
            $id = isset($data['id']) ? intval($data['id']) : (isset($data['id_documento_fe']) ? intval($data['id_documento_fe']) : 0);
            
            $sql = "SELECT pdf_url FROM tm_facturacion_electronica_documentos WHERE id_documento_fe = ?";
            $stm = $this->db->prepare($sql);
            $stm->execute(array($id));
            $doc = $stm->fetch(PDO::FETCH_ASSOC);
            
            if ($doc && $doc['pdf_url']) {
                header('Location: ' . $doc['pdf_url']);
                exit;
            } else {
                header('HTTP/1.0 404 Not Found');
                echo 'PDF no disponible';
            }
        } catch (Exception $e) {
            error_log("Error al descargar PDF: " . $e->getMessage());
            header('HTTP/1.0 500 Internal Server Error');
        }
    }
    
    /**
     * Descargar XML del documento desde facturafacil
     */
    public function descargar_xml($data)
    {
        try {
            $id = isset($data['id']) ? intval($data['id']) : (isset($data['id_documento_fe']) ? intval($data['id_documento_fe']) : 0);
            
            // Obtener documento con CUFE y UUID
            $sql = "SELECT d.*, 
                    COALESCE(s.serie, td.serie) as serie_electronica
                    FROM tm_facturacion_electronica_documentos d
                    LEFT JOIN tm_series_documentos s ON d.id_serie_documento = s.id_serie
                    LEFT JOIN tm_venta v ON d.id_venta = v.id_venta
                    LEFT JOIN tm_tipo_doc td ON v.id_tipo_doc = td.id_tipo_doc AND d.id_serie_documento IS NULL
                    WHERE d.id_documento_fe = ?";
            $stm = $this->db->prepare($sql);
            $stm->execute(array($id));
            $doc = $stm->fetch(PDO::FETCH_ASSOC);
            
            if (!$doc) {
                header('HTTP/1.0 404 Not Found');
                echo 'Documento no encontrado';
                exit;
            }
            
            // Si tenemos XML guardado localmente, usarlo primero
            if ($doc['xml_data']) {
                header('Content-Type: application/xml');
                header('Content-Disposition: attachment; filename="' . $doc['serie_electronica'] . '-' . str_pad($doc['fd_number'], 10, '0', STR_PAD_LEFT) . '.xml"');
                echo $doc['xml_data'];
                exit;
            }
            
            // Si no hay XML local pero tenemos CUFE y UUID, descargarlo de la API
            if (empty($doc['cufe']) || empty($doc['document_uuid'])) {
                header('HTTP/1.0 400 Bad Request');
                echo 'El documento no tiene CUFE o UUID. Debe estar autorizado primero.';
                exit;
            }
            
            // Obtener credenciales
            $stm_fe = $this->db->prepare("SELECT * FROM tm_facturacion_electronica WHERE id_facturacion = 1");
            $stm_fe->execute();
            $credenciales = $stm_fe->fetch(PDO::FETCH_ASSOC);
            
            if (!$credenciales || $credenciales['activo'] != 1) {
                header('HTTP/1.0 500 Internal Server Error');
                echo 'La facturación electrónica no está activa';
                exit;
            }
            
            // Determinar URL según ambiente
            $url_base = ($doc['environment'] == 1) 
                ? 'https://backend-api.facturafacil.com.pa'
                : 'https://backend-qa-api.facturafacil.com.pa';
            
            // Construir URL del endpoint
            $document_uuid = urlencode($doc['document_uuid']);
            $cufe = urlencode($doc['cufe']);
            $url_api = $url_base . '/api/pac/reception_fe/' . $document_uuid . '/download_xml/?cufe=' . $cufe;
            
            error_log("descargar_xml: URL API: " . $url_api);
            error_log("descargar_xml: Document UUID: " . $doc['document_uuid']);
            error_log("descargar_xml: CUFE: " . $doc['cufe']);
            
            // Realizar petición GET a la API
            $headers = array(
                'X-FF-Company: ' . $credenciales['ff_company_uuid'],
                'X-FF-API-Key: ' . $credenciales['ff_api_key'],
                'X-FF-Branch: ' . ($credenciales['ff_branch'] ?: ''),
                'Accept: application/xml, text/xml, */*'
            );
            
            $ch = curl_init();
            curl_setopt_array($ch, array(
                CURLOPT_URL => $url_api,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_FOLLOWLOCATION => true,
                CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
                CURLOPT_CUSTOMREQUEST => 'GET',
                CURLOPT_HTTPHEADER => $headers,
            ));
            
            $response = curl_exec($ch);
            $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $content_type = curl_getinfo($ch, CURLINFO_CONTENT_TYPE);
            $curl_error = curl_error($ch);
            curl_close($ch);
            
            error_log("descargar_xml: HTTP Code: " . $http_code);
            error_log("descargar_xml: Content-Type recibido: " . ($content_type ?: 'N/A'));
            error_log("descargar_xml: Response length: " . strlen($response));
            
            if ($curl_error) {
                error_log("Error CURL al descargar XML: " . $curl_error);
                header('HTTP/1.0 500 Internal Server Error');
                echo 'Error al descargar XML: ' . $curl_error;
                exit;
            }
            
            if ($http_code == 200) {
                // Guardar XML en la base de datos para futuras descargas
                if (!empty($response)) {
                    $sql_update = "UPDATE tm_facturacion_electronica_documentos SET xml_data = ? WHERE id_documento_fe = ?";
                    $stm_update = $this->db->prepare($sql_update);
                    $stm_update->execute(array($response, $id));
                }
                
                header('Content-Type: ' . ($content_type ?: 'application/xml'));
                header('Content-Disposition: attachment; filename="' . $doc['serie_electronica'] . '-' . str_pad($doc['fd_number'], 10, '0', STR_PAD_LEFT) . '.xml"');
                echo $response;
                exit;
            } else {
                $error_message = 'Error al descargar XML. Código HTTP: ' . $http_code;
                if ($http_code == 404) {
                    $error_message .= '. El documento no fue encontrado en facturafacil.';
                } elseif ($http_code == 406) {
                    $error_message .= '. El servidor no puede producir una respuesta en el formato solicitado.';
                }
                
                $response_decoded = json_decode($response, true);
                if ($response_decoded && isset($response_decoded['detail'])) {
                    $error_message .= ' Detalles: ' . $response_decoded['detail'];
                } elseif (!empty($response)) {
                    $error_message .= ' Respuesta: ' . substr($response, 0, 200);
                }
                
                error_log("Error al descargar XML - HTTP Code: " . $http_code . ", Response: " . $response);
                header('HTTP/1.0 ' . $http_code);
                echo $error_message;
                exit;
            }
            
        } catch (Exception $e) {
            error_log("Error al descargar XML: " . $e->getMessage());
            header('HTTP/1.0 500 Internal Server Error');
            echo 'Error: ' . $e->getMessage();
            exit;
        }
    }
    
    /**
     * Descargar CUFE del documento desde facturafacil
     */
    public function descargar_cafe($data)
    {
        try {
            $id = isset($data['id']) ? intval($data['id']) : (isset($data['id_documento_fe']) ? intval($data['id_documento_fe']) : 0);
            $copies = isset($data['copies']) ? intval($data['copies']) : 1; // Por defecto 1 copia
            $view = isset($data['view']) ? filter_var($data['view'], FILTER_VALIDATE_BOOLEAN) : false; // Por defecto descarga
            
            // Validar copias (máximo 5)
            if ($copies < 1) $copies = 1;
            if ($copies > 5) $copies = 5;
            
            // Obtener documento con CUFE y UUID
            $sql = "SELECT d.*, 
                    COALESCE(s.serie, td.serie) as serie_electronica
                    FROM tm_facturacion_electronica_documentos d
                    LEFT JOIN tm_series_documentos s ON d.id_serie_documento = s.id_serie
                    LEFT JOIN tm_venta v ON d.id_venta = v.id_venta
                    LEFT JOIN tm_tipo_doc td ON v.id_tipo_doc = td.id_tipo_doc AND d.id_serie_documento IS NULL
                    WHERE d.id_documento_fe = ?";
            $stm = $this->db->prepare($sql);
            $stm->execute(array($id));
            $doc = $stm->fetch(PDO::FETCH_ASSOC);
            
            if (!$doc) {
                header('HTTP/1.0 404 Not Found');
                echo 'Documento no encontrado';
                exit;
            }
            
            // Validar que tenga CUFE y UUID
            if (empty($doc['cufe'])) {
                header('HTTP/1.0 400 Bad Request');
                echo 'El documento no tiene CUFE. Debe estar autorizado primero.';
                exit;
            }
            
            if (empty($doc['document_uuid'])) {
                header('HTTP/1.0 400 Bad Request');
                echo 'El documento no tiene UUID. Debe estar autorizado primero.';
                exit;
            }
            
            // Obtener credenciales
            $stm_fe = $this->db->prepare("SELECT * FROM tm_facturacion_electronica WHERE id_facturacion = 1");
            $stm_fe->execute();
            $credenciales = $stm_fe->fetch(PDO::FETCH_ASSOC);
            
            if (!$credenciales || $credenciales['activo'] != 1) {
                header('HTTP/1.0 500 Internal Server Error');
                echo 'La facturación electrónica no está activa';
                exit;
            }
            
            // Determinar URL según ambiente
            $url_base = ($doc['environment'] == 1) 
                ? 'https://backend-api.facturafacil.com.pa'
                : 'https://backend-qa-api.facturafacil.com.pa';
            
            // Construir URL del endpoint
            // El CUFE es requerido como parámetro query
            $document_uuid = urlencode($doc['document_uuid']);
            $cufe = urlencode($doc['cufe']);
            
            // Construir parámetros de query
            $query_params = array(
                'cufe' => $cufe,
                'copies' => $copies
            );
            
            // Solo agregar view si es true (algunas APIs prefieren omitir parámetros false)
            if ($view) {
                $query_params['view'] = 'true';
            }
            
            $url_api = $url_base . '/api/pac/reception_fe/' . $document_uuid . '/download_cafe/?' . http_build_query($query_params);
            
            error_log("descargar_cafe: URL API: " . $url_api);
            error_log("descargar_cafe: Document UUID: " . $doc['document_uuid']);
            error_log("descargar_cafe: CUFE: " . $doc['cufe']);
            error_log("descargar_cafe: Copies: " . $copies);
            error_log("descargar_cafe: View: " . ($view ? 'true' : 'false'));
            
            // Realizar petición GET a la API
            // Para descargas de archivos PDF, usar Accept: */* o no enviarlo
            // El error 406 indica que el servidor no acepta application/pdf
            $headers = array(
                'X-FF-Company: ' . $credenciales['ff_company_uuid'],
                'X-FF-API-Key: ' . $credenciales['ff_api_key'],
                'X-FF-Branch: ' . ($credenciales['ff_branch'] ?: ''),
                'Accept: */*'
            );
            
            $ch = curl_init();
            curl_setopt_array($ch, array(
                CURLOPT_URL => $url_api,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_FOLLOWLOCATION => true,
                CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
                CURLOPT_CUSTOMREQUEST => 'GET',
                CURLOPT_HTTPHEADER => $headers,
            ));
            
            $response = curl_exec($ch);
            $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $content_type = curl_getinfo($ch, CURLINFO_CONTENT_TYPE);
            $curl_error = curl_error($ch);
            
            // Obtener headers de respuesta para debugging
            $response_headers = curl_getinfo($ch);
            
            curl_close($ch);
            
            error_log("descargar_cafe: HTTP Code: " . $http_code);
            error_log("descargar_cafe: Content-Type recibido: " . ($content_type ?: 'N/A'));
            error_log("descargar_cafe: Response length: " . strlen($response));
            error_log("descargar_cafe: Response (primeros 500 chars): " . substr($response, 0, 500));
            
            if ($curl_error) {
                error_log("Error CURL al descargar CUFE: " . $curl_error);
                header('HTTP/1.0 500 Internal Server Error');
                echo 'Error al descargar CUFE: ' . $curl_error;
                exit;
            }
            
            if ($http_code == 200) {
                // Si view es true, mostrar en el navegador, sino descargar
                if ($view) {
                    header('Content-Type: ' . ($content_type ?: 'application/pdf'));
                } else {
                    header('Content-Type: ' . ($content_type ?: 'application/pdf'));
                    header('Content-Disposition: attachment; filename="CUFE_' . $doc['serie_electronica'] . '-' . str_pad($doc['fd_number'], 10, '0', STR_PAD_LEFT) . '.pdf"');
                }
                echo $response;
                exit;
            } else {
                // Intentar parsear la respuesta como JSON si es un error
                $error_message = 'Error al descargar CUFE. Código HTTP: ' . $http_code;
                if ($http_code == 404) {
                    $error_message .= '. El documento no fue encontrado en facturafacil.';
                } elseif ($http_code == 406) {
                    $error_message .= '. El servidor no puede producir una respuesta en el formato solicitado.';
                }
                
                // Intentar obtener más información del error si la respuesta es JSON
                $response_decoded = json_decode($response, true);
                if ($response_decoded && isset($response_decoded['message'])) {
                    $error_message .= ' Detalles: ' . $response_decoded['message'];
                } elseif (!empty($response)) {
                    $error_message .= ' Respuesta: ' . substr($response, 0, 200);
                }
                
                error_log("Error al descargar CUFE - HTTP Code: " . $http_code . ", Response: " . $response);
                header('HTTP/1.0 ' . $http_code);
                echo $error_message;
                exit;
            }
            
        } catch (Exception $e) {
            error_log("Error al descargar CAFE: " . $e->getMessage());
            header('HTTP/1.0 500 Internal Server Error');
            echo 'Error: ' . $e->getMessage();
            exit;
        }
    }
    
    /**
     * Enviar documento por correo electrónico
     */
    public function enviar_email($data)
    {
        try {
            $id = isset($data['id_documento_fe']) ? intval($data['id_documento_fe']) : 0;
            $email = isset($data['email']) ? $data['email'] : '';
            
            if (empty($email)) {
                return array('success' => false, 'message' => 'Correo electrónico requerido');
            }
            
            // Obtener documento
            $sql = "SELECT d.*, 
                    COALESCE(s.serie, td.serie) as serie_electronica
                    FROM tm_facturacion_electronica_documentos d
                    LEFT JOIN tm_series_documentos s ON d.id_serie_documento = s.id_serie
                    LEFT JOIN tm_venta v ON d.id_venta = v.id_venta
                    LEFT JOIN tm_tipo_doc td ON v.id_tipo_doc = td.id_tipo_doc AND d.id_serie_documento IS NULL
                    WHERE d.id_documento_fe = ?";
            $stm = $this->db->prepare($sql);
            $stm->execute(array($id));
            $doc = $stm->fetch(PDO::FETCH_ASSOC);
            
            if (!$doc || !$doc['pdf_url']) {
                return array('success' => false, 'message' => 'Documento no encontrado o sin PDF disponible');
            }
            
            // Obtener credenciales
            $stm_fe = $this->db->prepare("SELECT * FROM tm_facturacion_electronica WHERE id_facturacion = 1");
            $stm_fe->execute();
            $credenciales = $stm_fe->fetch(PDO::FETCH_ASSOC);
            
            if (!$credenciales) {
                return array('success' => false, 'message' => 'Credenciales no configuradas');
            }
            
            // Determinar URL según ambiente
            $url_api = ($doc['environment'] == 1) 
                ? 'https://backend-api.facturafacil.com.pa/api/pac/reception_fe/' . $doc['document_uuid'] . '/send_email/'
                : 'https://backend-qa-api.facturafacil.com.pa/api/pac/reception_fe/' . $doc['document_uuid'] . '/send_email/';
            
            // Enviar a la API
            $ch = curl_init();
            curl_setopt_array($ch, array(
                CURLOPT_URL => $url_api,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_ENCODING => '',
                CURLOPT_MAXREDIRS => 10,
                CURLOPT_TIMEOUT => 30,
                CURLOPT_FOLLOWLOCATION => true,
                CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
                CURLOPT_CUSTOMREQUEST => 'POST',
                CURLOPT_POSTFIELDS => json_encode(array('email' => $email)),
                CURLOPT_HTTPHEADER => array(
                    'X-FF-Company: ' . $credenciales['ff_company_uuid'],
                    'X-FF-API-Key: ' . $credenciales['ff_api_key'],
                    'X-FF-Branch: ' . ($credenciales['ff_branch'] ?: ''),
                    'Content-Type: application/json',
                    'Accept: application/json'
                ),
            ));
            
            $response = curl_exec($ch);
            $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);
            
            if ($http_code == 200) {
                return array('success' => true, 'message' => 'Correo enviado correctamente');
            } else {
                $response_data = json_decode($response, true);
                $error_msg = isset($response_data['detail']) ? $response_data['detail'] : 'Error al enviar el correo';
                return array('success' => false, 'message' => $error_msg);
            }
            
        } catch (Exception $e) {
            error_log("Error en enviar_email: " . $e->getMessage());
            return array('success' => false, 'message' => $e->getMessage());
        }
    }
    
    /**
     * Convertir Nota de Venta a Boleta o Factura
     * 
     * @param array $data Datos con id_documento_fe, id_venta, nuevo_tipo_doc
     * @return array Resultado de la conversión
     */
    public function convertir_documento($data)
    {
        $log_file = 'debug_facturacion_electronica.log';
        $log_message = date('Y-m-d H:i:s') . " === CONVERTIR DOCUMENTO - INICIO ===\n";
        $log_message .= "Datos recibidos:\n";
        $log_message .= "  - id_documento_fe: " . (isset($data['id_documento_fe']) ? $data['id_documento_fe'] : 'NO DEFINIDO') . "\n";
        $log_message .= "  - id_venta: " . (isset($data['id_venta']) ? $data['id_venta'] : 'NO DEFINIDO') . "\n";
        $log_message .= "  - nuevo_tipo_doc: " . (isset($data['nuevo_tipo_doc']) ? $data['nuevo_tipo_doc'] : 'NO DEFINIDO') . "\n";
        file_put_contents($log_file, $log_message, FILE_APPEND);
        error_log($log_message);
        
        try {
            // Validar datos requeridos
            if (!isset($data['id_documento_fe']) || empty($data['id_documento_fe'])) {
                return array('success' => false, 'message' => 'ID de documento no proporcionado');
            }
            
            if (!isset($data['id_venta']) || empty($data['id_venta'])) {
                return array('success' => false, 'message' => 'ID de venta no proporcionado');
            }
            
            if (!isset($data['nuevo_tipo_doc']) || !in_array($data['nuevo_tipo_doc'], array('1', '2'))) {
                return array('success' => false, 'message' => 'Tipo de documento inválido. Debe ser 1 (Boleta) o 2 (Factura)');
            }
            
            $id_documento_fe = intval($data['id_documento_fe']);
            $id_venta = intval($data['id_venta']);
            $nuevo_tipo_doc = intval($data['nuevo_tipo_doc']); // 1=Boleta, 2=Factura
            
            // Obtener documento FE original
            $sql_doc = "SELECT * FROM tm_facturacion_electronica_documentos WHERE id_documento_fe = ?";
            $stm_doc = $this->db->prepare($sql_doc);
            $stm_doc->execute(array($id_documento_fe));
            $doc_original = $stm_doc->fetch(PDO::FETCH_ASSOC);
            
            if (!$doc_original) {
                return array('success' => false, 'message' => 'Documento no encontrado');
            }
            
            // Verificar que el documento sea de una Nota de Venta (tipo_doc = 3)
            $sql_venta = "SELECT id_tipo_doc, id_cliente FROM tm_venta WHERE id_venta = ?";
            $stm_venta = $this->db->prepare($sql_venta);
            $stm_venta->execute(array($id_venta));
            $venta_data = $stm_venta->fetch(PDO::FETCH_ASSOC);
            
            if (!$venta_data) {
                return array('success' => false, 'message' => 'Venta no encontrada');
            }
            
            if ($venta_data['id_tipo_doc'] != 3) {
                return array('success' => false, 'message' => 'Este documento no es una Nota de Venta. Solo se pueden convertir Notas de Venta.');
            }
            
            // Obtener datos del nuevo tipo de documento desde tm_tipo_doc
            $sql_tipodoc = "SELECT * FROM tm_tipo_doc WHERE id_tipo_doc = ? AND estado = 'a'";
            $stm_tipodoc = $this->db->prepare($sql_tipodoc);
            $stm_tipodoc->execute(array($nuevo_tipo_doc));
            $tipodoc_data = $stm_tipodoc->fetch(PDO::FETCH_ASSOC);
            
            if (!$tipodoc_data) {
                return array('success' => false, 'message' => 'Tipo de documento no encontrado o inactivo');
            }
            
            // Iniciar transacción
            $this->db->beginTransaction();
            
            try {
                // 1. Actualizar tipo de documento en la venta (y cliente si se proporciona uno nuevo)
                $nuevo_cliente_id = isset($data['nuevo_cliente_id']) && !empty($data['nuevo_cliente_id']) ? intval($data['nuevo_cliente_id']) : null;
                $cliente_original_id = isset($venta_data['id_cliente']) ? intval($venta_data['id_cliente']) : null;
                
                if ($nuevo_cliente_id && $nuevo_cliente_id != $cliente_original_id) {
                    // Actualizar tipo de documento y cliente
                    $sql_update_venta = "UPDATE tm_venta SET id_tipo_doc = ?, id_cliente = ? WHERE id_venta = ?";
                    $stm_update_venta = $this->db->prepare($sql_update_venta);
                    $stm_update_venta->execute(array($nuevo_tipo_doc, $nuevo_cliente_id, $id_venta));
                    
                    $log_message = date('Y-m-d H:i:s') . " ✅ Venta actualizada. Nuevo tipo_doc: $nuevo_tipo_doc, Nuevo cliente: $nuevo_cliente_id\n";
                    
                    // Obtener datos del nuevo cliente para actualizar el receptor en el documento FE
                    $sql_cliente = "SELECT * FROM v_clientes WHERE id_cliente = ?";
                    $stm_cliente = $this->db->prepare($sql_cliente);
                    $stm_cliente->execute(array($nuevo_cliente_id));
                    $nuevo_cliente_data = $stm_cliente->fetch(PDO::FETCH_ASSOC);
                } else {
                    // Solo actualizar tipo de documento
                    $sql_update_venta = "UPDATE tm_venta SET id_tipo_doc = ? WHERE id_venta = ?";
                    $stm_update_venta = $this->db->prepare($sql_update_venta);
                    $stm_update_venta->execute(array($nuevo_tipo_doc, $id_venta));
                    
                    $log_message = date('Y-m-d H:i:s') . " ✅ Venta actualizada. Nuevo tipo_doc: $nuevo_tipo_doc\n";
                    $nuevo_cliente_data = null;
                }
                
                file_put_contents($log_file, $log_message, FILE_APPEND);
                
                // 2. Preparar datos para nuevo documento FE
                $nuevo_numero_actual = intval($tipodoc_data['numero_actual']) + 1;
                $nuevo_correlativo = str_pad($nuevo_numero_actual, 10, '0', STR_PAD_LEFT);
                
                // Determinar tipo_documento_electronico según el nuevo tipo
                $nuevo_document_type = isset($tipodoc_data['tipo_documento']) ? strtoupper(trim($tipodoc_data['tipo_documento'])) : 'FAC';
                
                // 3. Crear nuevo documento FE con los mismos datos pero nuevo tipo
                $sql_nuevo_doc = "INSERT INTO tm_facturacion_electronica_documentos (
                    id_venta, id_pedido, id_serie_documento,
                    environment, issue_date, pos,
                    fd_number, document_type, sale_type, info, total, dest_country, payment_term,
                    receptor_type, receptor_name, receptor_ruc, receptor_ruc_type, 
                    receptor_address, receptor_email, receptor_dv, receptor_location,
                    status
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
                
                $fecha = date('Y-m-d H:i:s');
                $pos = str_pad(substr(preg_replace('/[^0-9]/', '', $tipodoc_data['pos'] ?: '001'), 0, 3), 3, '0', STR_PAD_LEFT);
                
                // Determinar datos del receptor (usar nuevo cliente si se proporcionó, sino usar los del documento original)
                $receptor_type = $doc_original['receptor_type'];
                $receptor_name = $doc_original['receptor_name'];
                $receptor_ruc = $doc_original['receptor_ruc'];
                $receptor_ruc_type = $doc_original['receptor_ruc_type'];
                $receptor_address = $doc_original['receptor_address'];
                $receptor_email = $doc_original['receptor_email'];
                $receptor_dv = $doc_original['receptor_dv'];
                $receptor_location = $doc_original['receptor_location'];
                
                // Si se proporcionó un nuevo cliente, actualizar datos del receptor
                if ($nuevo_cliente_data) {
                    $receptor_name = $nuevo_cliente_data['razon_social'] ?: $nuevo_cliente_data['nombre'];
                    $receptor_email = $nuevo_cliente_data['correo'] ?: null;
                    
                    // Determinar tipo de receptor según tipo de cliente
                    if ($nuevo_cliente_data['tipo_cliente'] == 2) {
                        // Contribuyente
                        $receptor_type = '01';
                        $receptor_ruc = $nuevo_cliente_data['ruc'];
                        $receptor_ruc_type = strlen($nuevo_cliente_data['ruc']) == 9 ? '1' : '2';
                        $receptor_address = $nuevo_cliente_data['direccion'];
                        $receptor_location = $nuevo_cliente_data['ubicacion'];
                    } else {
                        // Consumidor Final
                        $receptor_type = '02';
                        $receptor_ruc = null;
                        $receptor_ruc_type = null;
                        $receptor_dv = null;
                        $receptor_location = null;
                        $receptor_address = $nuevo_cliente_data['direccion'] ?: null;
                    }
                }
                
                $stm_nuevo_doc = $this->db->prepare($sql_nuevo_doc);
                $stm_nuevo_doc->execute(array(
                    $id_venta,
                    $doc_original['id_pedido'],
                    null, // id_serie_documento (null porque viene de tm_tipo_doc)
                    $doc_original['environment'],
                    $fecha,
                    $pos,
                    $nuevo_numero_actual,
                    $nuevo_document_type,
                    $doc_original['sale_type'],
                    $doc_original['info'],
                    $doc_original['total'],
                    $doc_original['dest_country'],
                    $doc_original['payment_term'],
                    $receptor_type,
                    $receptor_name,
                    $receptor_ruc,
                    $receptor_ruc_type,
                    $receptor_address,
                    $receptor_email,
                    $receptor_dv,
                    $receptor_location,
                    'pending' // status inicial
                ));
                
                $id_documento_nuevo = $this->db->lastInsertId();
                
                $log_message = date('Y-m-d H:i:s') . " ✅ Nuevo documento FE creado. ID: $id_documento_nuevo\n";
                file_put_contents($log_file, $log_message, FILE_APPEND);
                
                // 4. Copiar items del documento original
                $sql_items = "SELECT * FROM tm_facturacion_electronica_items WHERE id_documento_fe = ? ORDER BY line";
                $stm_items = $this->db->prepare($sql_items);
                $stm_items->execute(array($id_documento_fe));
                $items = $stm_items->fetchAll(PDO::FETCH_ASSOC);
                
                foreach ($items as $item) {
                    $sql_insert_item = "INSERT INTO tm_facturacion_electronica_items (
                        id_documento_fe, id_producto, line, price, mu, quantity, description,
                        internal_code, discount, gns, additional_info,
                        expiration_date, manufacturing_date,
                        medicine_batch_number, medicine_batch_quantity,
                        subtotal, total_taxes, total_item
                    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
                    
                    $stm_insert_item = $this->db->prepare($sql_insert_item);
                    $stm_insert_item->execute(array(
                        $id_documento_nuevo,
                        $item['id_producto'],
                        $item['line'],
                        $item['price'],
                        $item['mu'],
                        $item['quantity'],
                        $item['description'],
                        $item['internal_code'],
                        $item['discount'],
                        $item['gns'],
                        $item['additional_info'],
                        $item['expiration_date'],
                        $item['manufacturing_date'],
                        $item['medicine_batch_number'],
                        $item['medicine_batch_quantity'],
                        $item['subtotal'],
                        $item['total_taxes'],
                        $item['total_item']
                    ));
                    
                    $id_item_nuevo = $this->db->lastInsertId();
                    
                    // Copiar impuestos del item
                    $sql_taxes = "SELECT * FROM tm_facturacion_electronica_item_taxes WHERE id_item_fe = ?";
                    $stm_taxes = $this->db->prepare($sql_taxes);
                    $stm_taxes->execute(array($item['id_item_fe']));
                    $taxes = $stm_taxes->fetchAll(PDO::FETCH_ASSOC);
                    
                    foreach ($taxes as $tax) {
                        $sql_insert_tax = "INSERT INTO tm_facturacion_electronica_item_taxes (
                            id_item_fe, type, code, amount, rate
                        ) VALUES (?, ?, ?, ?, ?)";
                        
                        $stm_insert_tax = $this->db->prepare($sql_insert_tax);
                        $stm_insert_tax->execute(array(
                            $id_item_nuevo,
                            $tax['type'],
                            $tax['code'],
                            $tax['amount'],
                            $tax['rate']
                        ));
                    }
                }
                
                // 5. Copiar pagos del documento original
                $sql_payments = "SELECT * FROM tm_facturacion_electronica_payments WHERE id_documento_fe = ?";
                $stm_payments = $this->db->prepare($sql_payments);
                $stm_payments->execute(array($id_documento_fe));
                $payments = $stm_payments->fetchAll(PDO::FETCH_ASSOC);
                
                foreach ($payments as $payment) {
                    $sql_insert_payment = "INSERT INTO tm_facturacion_electronica_payments (
                        id_documento_fe, type, amount, description, code, reference
                    ) VALUES (?, ?, ?, ?, ?, ?)";
                    
                    $stm_insert_payment = $this->db->prepare($sql_insert_payment);
                    $stm_insert_payment->execute(array(
                        $id_documento_nuevo,
                        $payment['type'],
                        $payment['amount'],
                        $payment['description'],
                        $payment['code'],
                        $payment['reference']
                    ));
                }
                
                // 6. Actualizar número actual en tm_tipo_doc
                $sql_update_tipodoc = "UPDATE tm_tipo_doc SET numero_actual = ? WHERE id_tipo_doc = ?";
                $stm_update_tipodoc = $this->db->prepare($sql_update_tipodoc);
                $stm_update_tipodoc->execute(array($nuevo_numero_actual, $nuevo_tipo_doc));
                
                // Confirmar transacción
                $this->db->commit();
                
                $log_message = date('Y-m-d H:i:s') . " ✅ Conversión completada exitosamente. Nuevo documento ID: $id_documento_nuevo\n";
                $log_message .= "=== CONVERTIR DOCUMENTO - FIN EXITOSO ===\n";
                file_put_contents($log_file, $log_message, FILE_APPEND);
                error_log($log_message);
                
                return array(
                    'success' => true,
                    'message' => 'Documento convertido exitosamente',
                    'id_documento_nuevo' => $id_documento_nuevo,
                    'nuevo_tipo_doc' => $nuevo_tipo_doc,
                    'nuevo_document_type' => $nuevo_document_type,
                    'nuevo_numero' => $nuevo_correlativo
                );
                
            } catch (Exception $e) {
                // Rollback en caso de error
                $this->db->rollBack();
                throw $e;
            }
            
        } catch (Exception $e) {
            $log_message = date('Y-m-d H:i:s') . " ❌ Error en convertir_documento: " . $e->getMessage() . "\n";
            $log_message .= "=== CONVERTIR DOCUMENTO - FIN CON ERROR ===\n";
            file_put_contents($log_file, $log_message, FILE_APPEND);
            error_log($log_message);
            
            return array(
                'success' => false,
                'message' => 'Error al convertir el documento: ' . $e->getMessage()
            );
        }
    }
    
    /**
     * Emitir Nota de Crédito para una factura
     */
    public function emitir_nota_credito($data)
    {
        $log_file = __DIR__ . '/../debug_facturacion_electronica.log';
        $log_message = date('Y-m-d H:i:s') . " === EMITIR NOTA DE CRÉDITO - INICIO ===\n";
        $log_message .= "Datos recibidos:\n" . print_r($data, true) . "\n";
        file_put_contents($log_file, $log_message, FILE_APPEND);
        
        try {
            date_default_timezone_set($_SESSION["zona_horaria"]);
            $fecha = date("Y-m-d H:i:s");
            
            $id_documento_fe = isset($data['id_documento_fe']) ? intval($data['id_documento_fe']) : 0;
            $motivo = isset($data['motivo']) ? trim($data['motivo']) : '';
            
            if ($id_documento_fe <= 0) {
                return array(
                    'success' => false,
                    'message' => 'ID de documento inválido'
                );
            }
            
            // 1. Obtener datos del documento original
            $sql_doc = "SELECT d.*, 
                       COALESCE(s.serie, td.serie) as serie_electronica,
                       v.id_venta, v.id_cliente, v.id_tipo_doc as id_tipo_doc_venta
                       FROM tm_facturacion_electronica_documentos d
                       LEFT JOIN tm_series_documentos s ON d.id_serie_documento = s.id_serie
                       LEFT JOIN tm_venta v ON d.id_venta = v.id_venta
                       LEFT JOIN tm_tipo_doc td ON v.id_tipo_doc = td.id_tipo_doc AND d.id_serie_documento IS NULL
                       WHERE d.id_documento_fe = ?";
            
            $stm_doc = $this->db->prepare($sql_doc);
            $stm_doc->execute(array($id_documento_fe));
            $doc_original = $stm_doc->fetch(PDO::FETCH_ASSOC);
            
            if (!$doc_original) {
                return array(
                    'success' => false,
                    'message' => 'Documento original no encontrado'
                );
            }
            
            // Verificar que sea una factura (FAC)
            if ($doc_original['document_type'] != 'FAC') {
                return array(
                    'success' => false,
                    'message' => 'Solo se pueden emitir Notas de Crédito para Facturas (FAC)'
                );
            }
            
            // Verificar que esté autorizada o enviada
            if ($doc_original['status'] != 'authorized' && $doc_original['status'] != 'sent') {
                return array(
                    'success' => false,
                    'message' => 'Solo se pueden emitir Notas de Crédito para facturas autorizadas o enviadas'
                );
            }
            
            // 2. Verificar/crear tipo de documento NTC (Nota de Crédito)
            // Buscar si existe id_tipo_doc = 4 para Nota de Crédito
            $sql_tipo_ntc = "SELECT * FROM tm_tipo_doc WHERE id_tipo_doc = 4 AND estado = 'a'";
            $stm_tipo_ntc = $this->db->prepare($sql_tipo_ntc);
            $stm_tipo_ntc->execute();
            $tipo_ntc = $stm_tipo_ntc->fetch(PDO::FETCH_ASSOC);
            
            if (!$tipo_ntc) {
                // Crear tipo de documento NTC si no existe
                $sql_insert_tipo = "INSERT INTO tm_tipo_doc 
                    (id_tipo_doc, descripcion, serie, pos, numero, numero_inicial, numero_actual, numero_final, ambiente, estado, defecto)
                    VALUES (4, 'NOTA DE CRÉDITO', 'NC01', '001', 1, 1, 1, 9999999999, 2, 'a', '0')";
                $this->db->prepare($sql_insert_tipo)->execute();
                
                // Obtener el tipo creado
                $stm_tipo_ntc = $this->db->prepare($sql_tipo_ntc);
                $stm_tipo_ntc->execute();
                $tipo_ntc = $stm_tipo_ntc->fetch(PDO::FETCH_ASSOC);
            }
            
            // 3. Obtener número siguiente para NTC
            $nuevo_numero = $tipo_ntc['numero_actual'] + 1;
            $serie_ntc = $tipo_ntc['serie'];
            $pos_ntc = $tipo_ntc['pos'] ?: '001';
            
            // 4. Iniciar transacción
            $this->db->beginTransaction();
            
            // 5. Crear nuevo documento NTC con referencia al documento original
            // Preparar fecha del documento original para referred_fd_date
            $fecha_original = date('Y-m-d', strtotime($doc_original['fecha_creacion']));
            
            // Preparar número del documento original (usar CUFE si existe, sino fd_number)
            $referred_fd_number = $doc_original['cufe'] ?: str_pad($doc_original['fd_number'], 10, '0', STR_PAD_LEFT);
            
            // Motivo se guarda en info (la tabla no tiene columna motivo_anulacion)
            $info_ntc = $motivo ? ('Nota de crédito. Motivo: ' . $motivo) : 'Nota de crédito por anulación de factura.';
            
            $sql_insert_doc = "INSERT INTO tm_facturacion_electronica_documentos (
                id_venta, id_pedido, id_serie_documento, environment, pos, fd_number, document_type,
                total, receptor_type, receptor_name, receptor_ruc, receptor_email, receptor_address,
                receptor_location, receptor_dv, status, fecha_creacion, fecha_envio, 
                referred_fd_number, referred_fd_date, info
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
            
            $stm_insert_doc = $this->db->prepare($sql_insert_doc);
            $stm_insert_doc->execute(array(
                $doc_original['id_venta'],
                $doc_original['id_pedido'],
                null, // id_serie_documento null porque usamos tm_tipo_doc
                $tipo_ntc['ambiente'] ?: 2,
                $pos_ntc,
                $nuevo_numero,
                'NTC', // Tipo de documento: Nota de Crédito
                $doc_original['total'],
                $doc_original['receptor_type'],
                $doc_original['receptor_name'],
                $doc_original['receptor_ruc'],
                $doc_original['receptor_email'],
                $doc_original['receptor_address'],
                $doc_original['receptor_location'],
                $doc_original['receptor_dv'],
                'pending',
                $fecha,
                null,
                $referred_fd_number, // Número o CUFE del documento referenciado
                $fecha_original,     // Fecha del documento referenciado
                $info_ntc            // Motivo en info (no existe motivo_anulacion en la tabla)
            ));
            
            $id_documento_ntc = $this->db->lastInsertId();
            
            // 7. Copiar items del documento original
            $sql_items = "SELECT * FROM tm_facturacion_electronica_items WHERE id_documento_fe = ? ORDER BY line";
            $stm_items = $this->db->prepare($sql_items);
            $stm_items->execute(array($id_documento_fe));
            $items = $stm_items->fetchAll(PDO::FETCH_ASSOC);
            
            foreach ($items as $item) {
                $sql_insert_item = "INSERT INTO tm_facturacion_electronica_items (
                    id_documento_fe, id_producto, line, price, mu, quantity, description,
                    internal_code, discount, gns, additional_info,
                    expiration_date, manufacturing_date,
                    medicine_batch_number, medicine_batch_quantity,
                    subtotal, total_taxes, total_item
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
                
                $stm_insert_item = $this->db->prepare($sql_insert_item);
                $stm_insert_item->execute(array(
                    $id_documento_ntc,
                    $item['id_producto'],
                    $item['line'],
                    $item['price'],
                    $item['mu'],
                    $item['quantity'],
                    $item['description'],
                    $item['internal_code'],
                    $item['discount'],
                    $item['gns'],
                    $item['additional_info'],
                    $item['expiration_date'],
                    $item['manufacturing_date'],
                    $item['medicine_batch_number'],
                    $item['medicine_batch_quantity'],
                    $item['subtotal'],
                    $item['total_taxes'],
                    $item['total_item']
                ));
                
                $id_item_nuevo = $this->db->lastInsertId();
                
                // Copiar impuestos del item
                $sql_taxes = "SELECT * FROM tm_facturacion_electronica_item_taxes WHERE id_item_fe = ?";
                $stm_taxes = $this->db->prepare($sql_taxes);
                $stm_taxes->execute(array($item['id_item_fe']));
                $taxes = $stm_taxes->fetchAll(PDO::FETCH_ASSOC);
                
                foreach ($taxes as $tax) {
                    $sql_insert_tax = "INSERT INTO tm_facturacion_electronica_item_taxes (
                        id_item_fe, type, code, rate, amount
                    ) VALUES (?, ?, ?, ?, ?)";
                    
                    $this->db->prepare($sql_insert_tax)->execute(array(
                        $id_item_nuevo,
                        $tax['type'],
                        $tax['code'],
                        $tax['rate'],
                        $tax['amount']
                    ));
                }
            }
            
            // 8. Copiar payments del documento original
            $sql_payments = "SELECT * FROM tm_facturacion_electronica_payments WHERE id_documento_fe = ?";
            $stm_payments = $this->db->prepare($sql_payments);
            $stm_payments->execute(array($id_documento_fe));
            $payments = $stm_payments->fetchAll(PDO::FETCH_ASSOC);
            
            foreach ($payments as $payment) {
                // La tabla tiene id_tipo_pago, type, amount, description (no reference ni due_date)
                $sql_insert_payment = "INSERT INTO tm_facturacion_electronica_payments (
                    id_documento_fe, id_tipo_pago, type, amount, description
                ) VALUES (?, ?, ?, ?, ?)";
                
                $this->db->prepare($sql_insert_payment)->execute(array(
                    $id_documento_ntc,
                    isset($payment['id_tipo_pago']) ? $payment['id_tipo_pago'] : null,
                    $payment['type'],
                    $payment['amount'],
                    isset($payment['description']) ? $payment['description'] : null
                ));
            }
            
            // 9. Actualizar número actual del tipo de documento NTC
            $sql_update_numero = "UPDATE tm_tipo_doc SET numero_actual = ? WHERE id_tipo_doc = 4";
            $this->db->prepare($sql_update_numero)->execute(array($nuevo_numero));
            
            // 10. Commit transacción
            $this->db->commit();
            
            // 11. Enviar la NTC a la API (según documentación se envía como cualquier documento)
            $mensaje_final = 'Nota de Crédito emitida exitosamente.';
            $res_envio = $this->enviar_documento_fe(array('id_documento_fe' => $id_documento_ntc));
            if (!empty($res_envio['success'])) {
                $mensaje_final .= ' Enviada a facturafacil correctamente.';
            } else {
                $mensaje_final .= ' Guardada como pendiente de envío. ' . (isset($res_envio['message']) ? $res_envio['message'] : '');
            }
            
            $log_message = date('Y-m-d H:i:s') . " ✅ Nota de Crédito creada exitosamente. ID: $id_documento_ntc\n";
            $log_message .= "Documento original ID: $id_documento_fe\n";
            $log_message .= "Número NTC: $nuevo_numero\n";
            $log_message .= "=== EMITIR NOTA DE CRÉDITO - FIN EXITOSO ===\n";
            file_put_contents($log_file, $log_message, FILE_APPEND);
            
            return array(
                'success' => true,
                'message' => $mensaje_final,
                'id_documento_ntc' => $id_documento_ntc,
                'numero_ntc' => $nuevo_numero,
                'serie_ntc' => $serie_ntc,
                'enviada' => !empty($res_envio['success'])
            );
            
        } catch (Exception $e) {
            // Rollback en caso de error
            if ($this->db->inTransaction()) {
                $this->db->rollBack();
            }
            
            $log_message = date('Y-m-d H:i:s') . " ❌ Error en emitir_nota_credito: " . $e->getMessage() . "\n";
            $log_message .= "Trace: " . $e->getTraceAsString() . "\n";
            $log_message .= "=== EMITIR NOTA DE CRÉDITO - FIN CON ERROR ===\n";
            file_put_contents($log_file, $log_message, FILE_APPEND);
            error_log($log_message);
            
            return array(
                'success' => false,
                'message' => 'Error al emitir la Nota de Crédito: ' . $e->getMessage()
            );
        }
    }
}

